"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.endpointMeteringService = exports.EndpointMeteringService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/security-solution-plugin/common/endpoint/constants");
var _constants2 = require("../../constants");
var _product = require("../../../common/product");
var _metering = require("../constants/metering");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EndpointMeteringService {
  constructor() {
    (0, _defineProperty2.default)(this, "type", void 0);
    (0, _defineProperty2.default)(this, "tier", void 0);
    (0, _defineProperty2.default)(this, "getUsageRecords", async ({
      taskId,
      cloudSetup,
      esClient,
      abortController,
      lastSuccessfulReport,
      config,
      logger
    }) => {
      var _heartbeatsResponse$h, _heartbeatsResponse$h2, _cloudSetup$serverles;
      this.setType(config);
      if (!this.type) {
        return {
          records: []
        };
      }
      this.setTier(config);
      const heartbeatsResponse = await this.getHeartbeatsSince(esClient, abortController, lastSuccessfulReport);
      if (!(heartbeatsResponse !== null && heartbeatsResponse !== void 0 && (_heartbeatsResponse$h = heartbeatsResponse.hits) !== null && _heartbeatsResponse$h !== void 0 && (_heartbeatsResponse$h2 = _heartbeatsResponse$h.hits) !== null && _heartbeatsResponse$h2 !== void 0 && _heartbeatsResponse$h2.length)) {
        return {
          records: []
        };
      }
      const projectId = cloudSetup === null || cloudSetup === void 0 ? void 0 : (_cloudSetup$serverles = cloudSetup.serverless) === null || _cloudSetup$serverles === void 0 ? void 0 : _cloudSetup$serverles.projectId;
      if (!projectId) {
        logger.warn(`project id missing for records starting from ${lastSuccessfulReport.toISOString()}`);
      }
      const records = heartbeatsResponse.hits.hits.reduce((acc, {
        _source
      }) => {
        if (!_source) {
          return acc;
        }
        const {
          agent,
          event
        } = _source;
        const record = this.buildMeteringRecord({
          agentId: agent.id,
          timestampStr: event.ingested,
          taskId,
          projectId
        });
        return [...acc, record];
      }, []);
      const latestTimestamp = new Date(records[records.length - 1].usage_timestamp);
      const shouldRunAgain = heartbeatsResponse.hits.hits.length === _constants2.METERING_SERVICE_BATCH_SIZE;
      return {
        latestTimestamp,
        records,
        shouldRunAgain
      };
    });
  }
  async getHeartbeatsSince(esClient, abortController, since) {
    return esClient.search({
      index: _constants.ENDPOINT_HEARTBEAT_INDEX_PATTERN,
      sort: 'event.ingested',
      size: _constants2.METERING_SERVICE_BATCH_SIZE,
      query: {
        bool: {
          must: {
            range: {
              'event.ingested': {
                gt: since.toISOString()
              }
            }
          },
          should: [{
            term: {
              billable: true
            }
          }, {
            bool: {
              must_not: [{
                exists: {
                  field: 'billable'
                }
              }]
            }
          }],
          minimum_should_match: 1
        }
      }
    }, {
      signal: abortController.signal,
      ignore: [404]
    });
  }
  buildMeteringRecord({
    agentId,
    timestampStr,
    taskId,
    projectId
  }) {
    const timestamp = new Date(timestampStr);
    timestamp.setMinutes(0);
    timestamp.setSeconds(0);
    timestamp.setMilliseconds(0);
    const usageRecord = {
      // keep endpoint instead of this.type as id prefix so
      // we don't double count in the event of add-on changes
      id: `endpoint-${agentId}-${timestamp.toISOString()}`,
      usage_timestamp: timestampStr,
      creation_timestamp: timestampStr,
      usage: {
        // type postfix is used to determine the PLI to bill
        type: `${_metering.METERING_TASK.USAGE_TYPE_PREFIX}${this.type}`,
        period_seconds: _metering.METERING_TASK.SAMPLE_PERIOD_SECONDS,
        quantity: 1
      },
      source: {
        id: taskId,
        instance_group_id: projectId || _metering.METERING_TASK.MISSING_PROJECT_ID,
        metadata: {
          tier: this.tier
        }
      }
    };
    return usageRecord;
  }
  setType(config) {
    if (this.type) {
      return;
    }
    let hasCloudAddOn = false;
    let hasEndpointAddOn = false;
    config.productTypes.forEach(productType => {
      if (productType.product_line === _product.ProductLine.cloud) {
        hasCloudAddOn = true;
      }
      if (productType.product_line === _product.ProductLine.endpoint) {
        hasEndpointAddOn = true;
      }
    });
    if (hasEndpointAddOn) {
      this.type = _product.ProductLine.endpoint;
      return;
    }
    if (hasCloudAddOn) {
      this.type = `${_product.ProductLine.cloud}_${_product.ProductLine.endpoint}`;
    }
  }
  setTier(config) {
    if (this.tier) {
      return;
    }
    const product = config.productTypes.find(productType =>
    // tiers are always matching so either is fine
    productType.product_line === _product.ProductLine.endpoint || productType.product_line === _product.ProductLine.cloud);
    // default essentials is safe since we only reach tier if add-on exists
    this.tier = (product === null || product === void 0 ? void 0 : product.product_tier) || _product.ProductTier.essentials;
  }
}
exports.EndpointMeteringService = EndpointMeteringService;
const endpointMeteringService = exports.endpointMeteringService = new EndpointMeteringService();