"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ONE_MILLISECOND_AS_NANOSECONDS = exports.EventLoopDelayCircuitBreaker = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _perf_hooks = require("perf_hooks");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ONE_MILLISECOND_AS_NANOSECONDS = exports.ONE_MILLISECOND_AS_NANOSECONDS = 1_000_000;

/**
 * Configuration interface for Event Loop Delay Circuit Breaker.
 */

class EventLoopDelayCircuitBreaker extends _utils.BaseCircuitBreaker {
  constructor(config) {
    super();
    (0, _defineProperty2.default)(this, "loopMonitor", void 0);
    (0, _defineProperty2.default)(this, "fromTimestamp", void 0);
    (0, _defineProperty2.default)(this, "lastHistogram", {
      min: 0,
      max: 0,
      mean: 0,
      exceeds: 0,
      stddev: 0,
      fromTimestamp: '',
      lastUpdatedAt: '',
      percentiles: {
        50: 0,
        75: 0,
        95: 0,
        99: 0
      }
    });
    this.config = config;
    const monitor = (0, _perf_hooks.monitorEventLoopDelay)();
    monitor.enable();
    this.fromTimestamp = new Date();
    this.loopMonitor = monitor;
  }
  async validate() {
    const lastUpdated = new Date();
    this.loopMonitor.disable();
    const {
      min: minNs,
      max: maxNs,
      mean: meanNs,
      exceeds: exceedsNs,
      stddev: stddevNs
    } = this.loopMonitor;
    this.lastHistogram = {
      min: this.nsToMs(minNs),
      max: this.nsToMs(maxNs),
      mean: this.nsToMs(meanNs),
      exceeds: this.nsToMs(exceedsNs),
      stddev: this.nsToMs(stddevNs),
      fromTimestamp: this.fromTimestamp.toISOString(),
      lastUpdatedAt: lastUpdated.toISOString(),
      percentiles: {
        50: this.nsToMs(this.loopMonitor.percentile(50)),
        75: this.nsToMs(this.loopMonitor.percentile(75)),
        95: this.nsToMs(this.loopMonitor.percentile(95)),
        99: this.nsToMs(this.loopMonitor.percentile(99))
      }
    };
    this.loopMonitor.enable();
    if (this.lastHistogram.mean > this.config.thresholdMillis) {
      return this.failure(`Event loop delay mean ${this.lastHistogram.mean.toFixed(2)}ms exceeds threshold`);
    }
    return this.success();
  }
  stats() {
    return this.lastHistogram;
  }
  validationIntervalMs() {
    return this.config.validationIntervalMs;
  }
  nsToMs(metric) {
    return metric / ONE_MILLISECOND_AS_NANOSECONDS;
  }
}
exports.EventLoopDelayCircuitBreaker = EventLoopDelayCircuitBreaker;