"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.monitoringEntitySourceRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _constants = require("../../../../../../common/constants");
var _monitoring_entity_source = require("../../../../../../common/api/entity_analytics/privilege_monitoring/monitoring_entity_source/monitoring_entity_source.gen");
var _assert_advanced_setting_enabled = require("../../../utils/assert_advanced_setting_enabled");
var _constants2 = require("../../constants");
/* eslint-disable @kbn/eslint/require-license-header */

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const monitoringEntitySourceRoute = (router, logger, config) => {
  router.versioned.post({
    access: 'public',
    path: '/api/entity_analytics/monitoring/entity_source',
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: _monitoring_entity_source.CreateEntitySourceRequestBody
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      await (0, _assert_advanced_setting_enabled.assertAdvancedSettingsEnabled)(await context.core, _constants.ENABLE_PRIVILEGED_USER_MONITORING_SETTING);
      const secSol = await context.securitySolution;
      const client = secSol.getMonitoringEntitySourceDataClient();
      const body = await client.init(request.body);
      const privMonDataClient = await secSol.getPrivilegeMonitoringDataClient();
      const engineStatus = await privMonDataClient.getEngineStatus();
      try {
        if (engineStatus.status === _constants2.PRIVILEGE_MONITORING_ENGINE_STATUS.STARTED) {
          await privMonDataClient.scheduleNow();
        }
      } catch (e) {
        logger.warn(`[Privilege Monitoring] Error scheduling task, received ${e.message}`);
      }
      return response.ok({
        body
      });
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      logger.error(`Error creating monitoring entity source sync config: ${error.message}`);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: error.message
      });
    }
  });
  router.versioned.get({
    access: 'public',
    path: '/api/entity_analytics/monitoring/entity_source/{id}',
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        params: _monitoring_entity_source.GetEntitySourceRequestParams
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const secSol = await context.securitySolution;
      const client = secSol.getMonitoringEntitySourceDataClient();
      const body = await client.get(request.params.id);
      return response.ok({
        body
      });
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      logger.error(`Error getting monitoring entity source sync config: ${error.message}`);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: error.message
      });
    }
  });
  router.versioned.put({
    access: 'public',
    path: '/api/entity_analytics/monitoring/entity_source/{id}',
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: _monitoring_entity_source.UpdateEntitySourceRequestBody,
        params: _monitoring_entity_source.UpdateEntitySourceRequestParams
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const secSol = await context.securitySolution;
      const client = secSol.getMonitoringEntitySourceDataClient();
      const body = await client.update({
        ...request.body,
        id: request.params.id
      });
      const privMonDataClient = await secSol.getPrivilegeMonitoringDataClient();
      const engineStatus = await privMonDataClient.getEngineStatus();
      try {
        if (engineStatus.status === _constants2.PRIVILEGE_MONITORING_ENGINE_STATUS.STARTED) {
          await privMonDataClient.scheduleNow();
        }
      } catch (e) {
        logger.warn(`[Privilege Monitoring] Error scheduling task, received ${e.message}`);
      }
      return response.ok({
        body
      });
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      logger.error(`Error creating monitoring entity source sync config: ${error.message}`);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: error.message
      });
    }
  });
};
exports.monitoringEntitySourceRoute = monitoringEntitySourceRoute;