"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.importRulesRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _fp = require("lodash/fp");
var _path = require("path");
var _zodHelpers = require("@kbn/zod-helpers");
var _rule_management = require("../../../../../../../common/api/detection_engine/rule_management");
var _constants = require("../../../../../../../common/constants");
var _utils = require("../../../../routes/utils");
var _prebuilt_rule_assets_client = require("../../../../prebuilt_rules/logic/rule_assets/prebuilt_rule_assets_client");
var _import_rule_action_connectors = require("../../../logic/import/action_connectors/import_rule_action_connectors");
var _validate_rule_actions = require("../../../logic/import/action_connectors/validate_rule_actions");
var _rule_source_importer = require("../../../logic/import/rule_source_importer");
var _import_rules = require("../../../logic/import/import_rules");
var _create_promise_from_rule_import_stream = require("../../../logic/import/create_promise_from_rule_import_stream");
var _import_rule_exceptions = require("../../../logic/import/import_rule_exceptions");
var _utils2 = require("../../../logic/import/utils");
var _utils3 = require("../../../utils/utils");
var _timeouts = require("../../timeouts");
var _prebuilt_rule_objects_client = require("../../../../prebuilt_rules/logic/rule_objects/prebuilt_rule_objects_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CHUNK_PARSED_OBJECT_SIZE = 50;
const importRulesRoute = (router, config, logger) => {
  router.versioned.post({
    access: 'public',
    path: _constants.DETECTION_ENGINE_RULES_IMPORT_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      body: {
        maxBytes: config.maxRuleImportPayloadBytes,
        output: 'stream'
      },
      timeout: {
        idleSocket: _timeouts.RULE_MANAGEMENT_IMPORT_EXPORT_SOCKET_TIMEOUT_MS
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_management.ImportRulesRequestQuery),
        body: _configSchema.schema.any() // validation on file object is accomplished later in the handler.
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      var _ctx$lists;
      const ctx = await context.resolve(['core', 'securitySolution', 'alerting', 'actions', 'lists', 'licensing']);
      const rulesClient = await ctx.alerting.getRulesClient();
      const detectionRulesClient = ctx.securitySolution.getDetectionRulesClient();
      const actionsClient = ctx.actions.getActionsClient();
      const actionSOClient = ctx.core.savedObjects.getClient({
        includedHiddenTypes: ['action']
      });
      const actionsImporter = ctx.core.savedObjects.getImporter(actionSOClient);
      const savedObjectsClient = ctx.core.savedObjects.client;
      const exceptionsClient = (_ctx$lists = ctx.lists) === null || _ctx$lists === void 0 ? void 0 : _ctx$lists.getExceptionListClient();
      const {
        filename
      } = request.body.file.hapi;
      const fileExtension = (0, _path.extname)(filename).toLowerCase();
      if (fileExtension !== '.ndjson') {
        return siemResponse.error({
          statusCode: 400,
          body: `Invalid file extension ${fileExtension}`
        });
      }
      const objectLimit = config.maxRuleImportExportSize;

      // parse file to separate out exceptions from rules
      const [{
        exceptions,
        rules,
        actionConnectors
      }] = await (0, _create_promise_from_rule_import_stream.createPromiseFromRuleImportStream)({
        stream: request.body.file,
        objectLimit
      });

      // import exceptions, includes validation
      const {
        errors: exceptionsErrors,
        successCount: exceptionsSuccessCount,
        success: exceptionsSuccess
      } = await (0, _import_rule_exceptions.importRuleExceptions)({
        exceptions,
        exceptionsClient,
        overwrite: request.query.overwrite_exceptions,
        maxExceptionsImportSize: objectLimit
      });
      // report on duplicate rules
      const [duplicateIdErrors, rulesToImportOrErrors] = (0, _utils3.getTupleDuplicateErrorsAndUniqueRules)(rules, request.query.overwrite);

      // import actions-connectors
      const {
        successCount: actionConnectorSuccessCount,
        success: actionConnectorSuccess,
        warnings: actionConnectorWarnings,
        errors: actionConnectorErrors
      } = await (0, _import_rule_action_connectors.importRuleActionConnectors)({
        actionConnectors,
        actionsImporter,
        overwrite: request.query.overwrite_action_connectors
      });
      const migratedRulesToImportOrErrors = await (0, _utils3.migrateLegacyActionsIds)(rulesToImportOrErrors, actionSOClient, actionsClient);
      const ruleSourceImporter = (0, _rule_source_importer.createRuleSourceImporter)({
        context: ctx.securitySolution,
        prebuiltRuleAssetsClient: (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(savedObjectsClient),
        prebuiltRuleObjectsClient: (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(rulesClient),
        logger
      });
      const [parsedRules, parsedRuleErrors] = (0, _fp.partition)(_utils2.isRuleToImport, migratedRulesToImportOrErrors);

      // After importing the actions and migrating action IDs on rules to import,
      // validate that all actions referenced by rules exist
      // Filter out rules that reference non-existent actions
      const {
        validatedActionRules,
        missingActionErrors
      } = await (0, _validate_rule_actions.validateRuleActions)({
        actionsClient,
        rules: parsedRules
      });
      const ruleChunks = (0, _fp.chunk)(CHUNK_PARSED_OBJECT_SIZE, validatedActionRules);
      const importRuleResponse = await (0, _import_rules.importRules)({
        ruleChunks,
        overwriteRules: request.query.overwrite,
        allowMissingConnectorSecrets: !!actionConnectors.length,
        ruleSourceImporter,
        detectionRulesClient
      });
      const parseErrors = parsedRuleErrors.map(error => (0, _utils.createBulkErrorObject)({
        statusCode: 400,
        message: error.message
      }));
      const importErrors = importRuleResponse.filter(_utils.isBulkError);
      const errors = [...parseErrors, ...duplicateIdErrors, ...importErrors, ...missingActionErrors];
      const successes = importRuleResponse.filter(resp => {
        if ((0, _utils.isImportRegular)(resp)) {
          return resp.status_code === 200;
        } else {
          return false;
        }
      });
      const importRulesResponse = {
        success: errors.length === 0,
        success_count: successes.length,
        rules_count: rules.length,
        errors,
        exceptions_errors: exceptionsErrors,
        exceptions_success: exceptionsSuccess,
        exceptions_success_count: exceptionsSuccessCount,
        action_connectors_success: actionConnectorSuccess,
        action_connectors_success_count: actionConnectorSuccessCount,
        action_connectors_errors: actionConnectorErrors,
        action_connectors_warnings: actionConnectorWarnings
      };
      return response.ok({
        body: _rule_management.ImportRulesResponse.parse(importRulesResponse)
      });
    } catch (err) {
      logger.error(`importRulesRoute: Caught error:`, err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.importRulesRoute = importRulesRoute;