"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useWithArtifactListData = void 0;
var _react = require("react");
var _reactQuery = require("@kbn/react-query");
var _securitysolutionHookUtils = require("@kbn/securitysolution-hook-utils");
var _constants = require("../../../common/constants");
var _use_url_params = require("../../../hooks/use_url_params");
var _use_kuery_from_exceptions_search_filter = require("./use_kuery_from_exceptions_search_filter");
var _artifacts = require("../../../hooks/artifacts");
var _use_url_pagination = require("../../../hooks/use_url_pagination");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useWithArtifactListData = (apiClient, searchableFields) => {
  const isMounted = (0, _securitysolutionHookUtils.useIsMounted)();
  const {
    urlParams: {
      filter,
      includedPolicies
    }
  } = (0, _use_url_params.useUrlParams)();
  const {
    pagination: {
      page,
      pageSize
    }
  } = (0, _use_url_pagination.useUrlPagination)();

  // Used to determine if the `does data exist` check should be done.
  const kuery = (0, _use_kuery_from_exceptions_search_filter.useKueryFromExceptionsSearchFilter)(filter, searchableFields, includedPolicies);
  const {
    data: doesDataExist,
    isFetching: isLoadingDataExists,
    refetch: checkIfDataExists
  } = (0, _reactQuery.useQuery)(['does-data-exists', apiClient], async () => apiClient.hasData(), {
    enabled: true,
    keepPreviousData: true,
    refetchOnWindowFocus: false
  });
  const [uiPagination, setUiPagination] = (0, _react.useState)({
    totalItemCount: 0,
    pageSize,
    pageSizeOptions: [..._constants.MANAGEMENT_PAGE_SIZE_OPTIONS],
    pageIndex: page - 1
  });
  const [isPageInitializing, setIsPageInitializing] = (0, _react.useState)(true);
  const listDataRequest = (0, _artifacts.useListArtifact)(apiClient, {
    page,
    perPage: pageSize,
    filter,
    policies: includedPolicies ? includedPolicies.split(',') : []
  }, searchableFields);
  const {
    data: listData,
    isFetching: isLoadingListData,
    error: listDataError,
    isSuccess: isSuccessListData
  } = listDataRequest;

  // Once we know if data exists, update the page initializing state.
  // This should only ever happen at most once;
  (0, _react.useEffect)(() => {
    if (isMounted()) {
      if (isPageInitializing && !isLoadingDataExists) {
        setIsPageInitializing(false);
      }
    }
  }, [isLoadingDataExists, isMounted, isPageInitializing]);

  // Update the uiPagination once the query succeeds
  (0, _react.useEffect)(() => {
    if (isMounted() && listData && !isLoadingListData && isSuccessListData) {
      setUiPagination(prevState => {
        return {
          ...prevState,
          pageIndex: listData.page - 1,
          pageSize: listData.per_page,
          totalItemCount: listData.total
        };
      });
    }
  }, [isLoadingListData, isMounted, isSuccessListData, listData]);

  // Keep the `doesDataExist` updated if we detect that list data result total is zero.
  // Anytime:
  //      1. the list data total is 0
  //      2. and page is 1
  //      3. and filter is empty
  //      4. and doesDataExists is `true`
  //  >> check if data exists again
  // OR, Anytime:
  //      1. `doesDataExists` is `false`,
  //      2. and page is 1
  //      3. and filter is empty
  //      4. the list data total is > 0
  //  >> Check if data exists again (which should return true
  (0, _react.useEffect)(() => {
    if (isMounted() && !isLoadingListData && !isLoadingDataExists && !listDataError && String(page) === '1' && !kuery && (
    // flow when there the last item on the list gets deleted,
    // and list goes back to being empty
    listData && listData.total === 0 && doesDataExist ||
    // Flow when the list starts off empty and the first item is added
    listData && listData.total > 0 && !doesDataExist)) {
      checkIfDataExists();
    }
  }, [checkIfDataExists, doesDataExist, filter, includedPolicies, isLoadingDataExists, isLoadingListData, isMounted, kuery, listData, listDataError, page]);
  return (0, _react.useMemo)(() => ({
    isPageInitializing,
    doesDataExist: doesDataExist !== null && doesDataExist !== void 0 ? doesDataExist : false,
    uiPagination,
    ...listDataRequest
  }), [doesDataExist, isPageInitializing, listDataRequest, uiPagination]);
};
exports.useWithArtifactListData = useWithArtifactListData;