"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enableFleetServerIfNecessary = void 0;
var _repoInfo = require("@kbn/repo-info");
var _uuid = require("uuid");
var _common = require("@kbn/fleet-plugin/common");
var _common2 = require("@kbn/spaces-plugin/common");
var _index_fleet_agent = require("./index_fleet_agent");
var _format_axios_error = require("../format_axios_error");
var _usage_tracker = require("./usage_tracker");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Will ensure that at least one fleet server is present in the `.fleet-agents` index. This will
 * enable the `Agent` section of kibana Fleet to be displayed. We skip on serverless because
 * Fleet Server agents are not checked against there.
 *
 * @param esClient
 * @param kbnClient
 * @param log
 * @param version
 */
const enableFleetServerIfNecessary = exports.enableFleetServerIfNecessary = _usage_tracker.usageTracker.track('enableFleetServerIfNecessary', async (esClient, isServerless = false, kbnClient, log = (0, _utils.createToolingLogger)(), version = _repoInfo.kibanaPackageJson.version) => {
  const activeSpaceId = await (0, _utils.fetchActiveSpaceId)(kbnClient);
  const agentPolicy = await getOrCreateFleetServerAgentPolicy(kbnClient, activeSpaceId, log);
  if (!isServerless && !(await hasFleetServerAgent(esClient, agentPolicy.id, activeSpaceId, log))) {
    var _agentPolicy$space_id;
    log.debug(`Indexing a new fleet server agent`);
    const lastCheckin = new Date();
    const agentVersion = version;
    lastCheckin.setFullYear(lastCheckin.getFullYear() + 1);
    const indexedAgent = await (0, _index_fleet_agent.indexFleetServerAgent)(esClient, log, {
      policy_id: agentPolicy.id,
      agent: {
        version: agentVersion
      },
      last_checkin_status: 'online',
      last_checkin: lastCheckin.toISOString(),
      namespaces: (_agentPolicy$space_id = agentPolicy.space_ids) !== null && _agentPolicy$space_id !== void 0 ? _agentPolicy$space_id : [activeSpaceId]
    });
    log.verbose(`New fleet server agent indexed:\n${JSON.stringify(indexedAgent, null, 2)}`);
  } else {
    log.debug(`Nothing to do. A Fleet Server agent is already registered with Fleet`);
  }
});
const getOrCreateFleetServerAgentPolicy = async (kbnClient, spaceId, log = (0, _utils.createToolingLogger)()) => {
  const packagePolicies = await kbnClient.request({
    method: 'GET',
    headers: {
      'elastic-api-version': '2023-10-31'
    },
    path: _common.packagePolicyRouteService.getListPath(),
    query: {
      perPage: 1,
      kuery: `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name: "${_common.FLEET_SERVER_PACKAGE}"`
    }
  }).catch(_format_axios_error.catchAxiosErrorFormatAndThrow);
  if (packagePolicies.data.items[0]) {
    log.debug(`Found an existing Fleet Server package policy [${packagePolicies.data.items[0].id}] - fetching associated agent policy`);
    log.verbose(JSON.stringify(packagePolicies.data, null, 2));
    return kbnClient.request({
      headers: {
        'elastic-api-version': '2023-10-31'
      },
      method: 'GET',
      path: _common.agentPolicyRouteService.getInfoPath(packagePolicies.data.items[0].policy_ids[0])
    }).catch(_format_axios_error.catchAxiosErrorFormatAndThrow).then(response => {
      log.debug(`Returning existing Fleet Server agent policy [${response.data.item.id}]`);
      log.verbose(`Existing agent policy for Fleet Server:\n${JSON.stringify(response.data.item, null, 2)}`);
      return response.data.item;
    });
  }
  log.debug(`Creating a new fleet server agent policy`);
  const policy = {
    name: `Fleet Server policy (${Math.random().toString(32).substring(2)})`,
    id: (0, _uuid.v4)(),
    description: `Created by CLI Tool via: ${__filename}`,
    namespace: spaceId !== null && spaceId !== void 0 ? spaceId : _common2.DEFAULT_SPACE_ID,
    monitoring_enabled: [],
    // This will ensure the Fleet Server integration policy
    // is also created and added to the agent policy
    has_fleet_server: true
  };
  log.verbose(`New policy:\n${JSON.stringify(policy, null, 2)}`);

  // create new Fleet Server agent policy
  return kbnClient.request({
    method: 'POST',
    path: _common.AGENT_POLICY_API_ROUTES.CREATE_PATTERN,
    headers: {
      'elastic-api-version': '2023-10-31'
    },
    body: policy
  }).then(response => {
    log.verbose(`No fleet server agent policy found. Created a new one:\n${JSON.stringify(response.data.item, null, 2)}`);
    return response.data.item;
  }).catch(_format_axios_error.catchAxiosErrorFormatAndThrow);
};
const hasFleetServerAgent = async (esClient, fleetServerAgentPolicyId, spaceId, log = (0, _utils.createToolingLogger)()) => {
  const query = {
    bool: {
      filter: [{
        term: {
          policy_id: fleetServerAgentPolicyId
        }
      }, ...(spaceId ? [{
        term: {
          namespaces: spaceId
        }
      }] : [])]
    }
  };
  const searchResponse = await esClient.search({
    index: _common.AGENTS_INDEX,
    ignore_unavailable: true,
    rest_total_hits_as_int: true,
    size: 1,
    query
  }, {
    ignore: [404]
  }).catch(_utils.wrapErrorAndRejectPromise);
  log.verbose(`Search for a fleet server agent with query:\n${JSON.stringify(query, null, 2)}\nreturn:\n ${fleetServerAgentPolicyId}]\n${JSON.stringify(searchResponse, null, 2)}`);
  return Boolean(searchResponse === null || searchResponse === void 0 ? void 0 : searchResponse.hits.total);
};