"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTemplateInputs = getTemplateInputs;
exports.templatePackagePolicyToFullInputStreams = void 0;
var _lodash = require("lodash");
var _jsYaml = require("js-yaml");
var _yaml = _interopRequireDefault(require("yaml"));
var _services = require("../../../../common/services");
var _package_to_package_policy = require("../../../../common/services/package_to_package_policy");
var _package_policy = require("../../package_policy");
var _app_context = require("../../app_context");
var _full_agent_policy_to_yaml = require("../../../../common/services/full_agent_policy_to_yaml");
var _package_policies_to_agent_inputs = require("../../agent_policies/package_policies_to_agent_inputs");
var _ = require(".");
var _get = require("./get");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Function based off storedPackagePolicyToAgentInputs, it only creates the `streams` section instead of the FullAgentPolicyInput
const templatePackagePolicyToFullInputStreams = (packagePolicyInputs, inputAndStreamsIdsMap) => {
  const fullInputsStreams = [];
  if (!packagePolicyInputs || packagePolicyInputs.length === 0) return fullInputsStreams;
  packagePolicyInputs.forEach(input => {
    const streamsIdsMap = new Map();
    const inputId = input.policy_template ? `${input.policy_template}-${input.type}` : `${input.type}`;
    const fullInputStream = {
      // @ts-ignore-next-line the following id is actually one level above the one in fullInputStream, but the linter thinks it gets overwritten
      id: inputId,
      type: input.type,
      ...(0, _package_policies_to_agent_inputs.getFullInputStreams)(input, true, streamsIdsMap)
    };
    inputAndStreamsIdsMap === null || inputAndStreamsIdsMap === void 0 ? void 0 : inputAndStreamsIdsMap.set(fullInputStream.id, {
      originalId: inputId,
      streams: streamsIdsMap
    });

    // deeply merge the input.config values with the full policy input stream
    (0, _lodash.merge)(fullInputStream, Object.entries(input.config || {}).reduce((acc, [key, {
      value
    }]) => {
      acc[key] = value;
      return acc;
    }, {}));
    fullInputsStreams.push(fullInputStream);
  });
  return fullInputsStreams;
};
exports.templatePackagePolicyToFullInputStreams = templatePackagePolicyToFullInputStreams;
async function getTemplateInputs(soClient, pkgName, pkgVersion, format, prerelease, ignoreUnverified) {
  const packageInfo = await (0, _.getPackageInfo)({
    savedObjectsClient: soClient,
    pkgName,
    pkgVersion,
    prerelease,
    ignoreUnverified
  });
  const emptyPackagePolicy = (0, _package_to_package_policy.packageToPackagePolicy)(packageInfo, '');
  const inputsWithStreamIds = (0, _package_policy.getInputsWithStreamIds)(emptyPackagePolicy, undefined, true);
  const indexedInputsAndStreams = buildIndexedPackage(packageInfo);
  if (format === 'yml') {
    // Add a placeholder <VAR_NAME> to all variables without default value
    for (const inputWithStreamIds of inputsWithStreamIds) {
      const inputId = inputWithStreamIds.policy_template ? `${inputWithStreamIds.policy_template}-${inputWithStreamIds.type}` : inputWithStreamIds.type;
      const packageInput = indexedInputsAndStreams[inputId];
      if (!packageInput) {
        continue;
      }
      for (const [inputVarKey, inputVarValue] of Object.entries((_inputWithStreamIds$v = inputWithStreamIds.vars) !== null && _inputWithStreamIds$v !== void 0 ? _inputWithStreamIds$v : {})) {
        var _inputWithStreamIds$v, _packageInput$vars;
        const varDef = (_packageInput$vars = packageInput.vars) === null || _packageInput$vars === void 0 ? void 0 : _packageInput$vars.find(_varDef => _varDef.name === inputVarKey);
        if (varDef) {
          addPlaceholderIfNeeded(varDef, inputVarValue);
        }
      }
      for (const stream of inputWithStreamIds.streams) {
        const packageStream = packageInput.streams[stream.id];
        if (!packageStream) {
          continue;
        }
        for (const [streamVarKey, streamVarValue] of Object.entries((_stream$vars = stream.vars) !== null && _stream$vars !== void 0 ? _stream$vars : {})) {
          var _stream$vars, _packageStream$vars;
          const varDef = (_packageStream$vars = packageStream.vars) === null || _packageStream$vars === void 0 ? void 0 : _packageStream$vars.find(_varDef => _varDef.name === streamVarKey);
          if (varDef) {
            addPlaceholderIfNeeded(varDef, streamVarValue);
          }
        }
      }
    }
  }
  const assetsMap = await (0, _get.getAgentTemplateAssetsMap)({
    logger: _app_context.appContextService.getLogger(),
    packageInfo,
    savedObjectsClient: soClient,
    ignoreUnverified
  });
  const compiledInputs = await (0, _package_policy._compilePackagePolicyInputs)(packageInfo, emptyPackagePolicy.vars || {}, inputsWithStreamIds, assetsMap);
  const packagePolicyWithInputs = {
    ...emptyPackagePolicy,
    inputs: compiledInputs
  };
  const inputIdsDestinationMap = new Map();
  const inputs = templatePackagePolicyToFullInputStreams(packagePolicyWithInputs.inputs, inputIdsDestinationMap);
  if (format === 'json') {
    return {
      inputs
    };
  } else if (format === 'yml') {
    const yaml = (0, _jsYaml.dump)({
      inputs
    }, {
      skipInvalid: true,
      sortKeys: _full_agent_policy_to_yaml._sortYamlKeys
    });
    return addCommentsToYaml(yaml, buildIndexedPackage(packageInfo), inputIdsDestinationMap);
  }
  return {
    inputs: []
  };
}
function getPlaceholder(varDef) {
  return `<${varDef.name.toUpperCase()}>`;
}
function addPlaceholderIfNeeded(varDef, varValue) {
  const placeHolder = `<${varDef.name.toUpperCase()}>`;
  if (varDef && !varValue.value && varDef.type !== 'yaml') {
    varValue.value = placeHolder;
  } else if (varDef && varValue.value && varValue.value.length === 0 && varDef.type === 'text') {
    varValue.value = [placeHolder];
  }
}
function buildIndexedPackage(packageInfo) {
  var _packageInfo$policy_t, _packageInfo$policy_t2;
  return (_packageInfo$policy_t = (_packageInfo$policy_t2 = packageInfo.policy_templates) === null || _packageInfo$policy_t2 === void 0 ? void 0 : _packageInfo$policy_t2.reduce((inputsAcc, policyTemplate) => {
    const inputs = (0, _services.getNormalizedInputs)(policyTemplate);
    inputs.forEach(packageInput => {
      const inputId = `${policyTemplate.name}-${packageInput.type}`;
      const streams = (0, _package_to_package_policy.getStreamsForInputType)(packageInput.type, packageInfo, (0, _services.isIntegrationPolicyTemplate)(policyTemplate) && policyTemplate.data_streams ? policyTemplate.data_streams : []).reduce((acc, stream) => {
        const streamId = `${packageInput.type}-${stream.data_stream.dataset}`;
        acc[streamId] = {
          ...stream
        };
        return acc;
      }, {});
      inputsAcc[inputId] = {
        ...packageInput,
        streams
      };
    });
    return inputsAcc;
  }, {})) !== null && _packageInfo$policy_t !== void 0 ? _packageInfo$policy_t : {};
}
function addCommentsToYaml(yaml, packageIndexInputAndStreams, inputIdsDestinationMap) {
  const doc = _yaml.default.parseDocument(yaml);
  // Add input and streams comments
  const yamlInputs = doc.get('inputs');
  if (_yaml.default.isCollection(yamlInputs)) {
    yamlInputs.items.forEach(inputItem => {
      var _inputIdsDestinationM, _inputIdsDestinationM2;
      if (!_yaml.default.isMap(inputItem)) {
        return;
      }
      const inputIdNode = inputItem.get('id', true);
      if (!_yaml.default.isScalar(inputIdNode)) {
        return;
      }
      const inputId = (_inputIdsDestinationM = (_inputIdsDestinationM2 = inputIdsDestinationMap.get(inputIdNode.value)) === null || _inputIdsDestinationM2 === void 0 ? void 0 : _inputIdsDestinationM2.originalId) !== null && _inputIdsDestinationM !== void 0 ? _inputIdsDestinationM : inputIdNode.value;
      const pkgInput = packageIndexInputAndStreams[inputId];
      if (pkgInput) {
        var _pkgInput$vars;
        inputItem.commentBefore = ` ${pkgInput.title}${pkgInput.description ? `: ${pkgInput.description}` : ''}`;
        commentVariablesInYaml(inputItem, (_pkgInput$vars = pkgInput.vars) !== null && _pkgInput$vars !== void 0 ? _pkgInput$vars : []);
        const yamlStreams = inputItem.get('streams');
        if (!_yaml.default.isCollection(yamlStreams)) {
          return;
        }
        yamlStreams.items.forEach(streamItem => {
          if (!_yaml.default.isMap(streamItem)) {
            return;
          }
          const streamIdNode = streamItem.get('id', true);
          if (_yaml.default.isScalar(streamIdNode)) {
            var _inputIdsDestinationM3, _inputIdsDestinationM4, _inputIdsDestinationM5;
            const streamId = (_inputIdsDestinationM3 = (_inputIdsDestinationM4 = inputIdsDestinationMap.get(inputIdNode.value)) === null || _inputIdsDestinationM4 === void 0 ? void 0 : (_inputIdsDestinationM5 = _inputIdsDestinationM4.streams) === null || _inputIdsDestinationM5 === void 0 ? void 0 : _inputIdsDestinationM5.get(streamIdNode.value)) !== null && _inputIdsDestinationM3 !== void 0 ? _inputIdsDestinationM3 : streamIdNode.value;
            const pkgStream = pkgInput.streams[streamId];
            if (pkgStream) {
              var _pkgStream$vars;
              streamItem.commentBefore = ` ${pkgStream.title}${pkgStream.description ? `: ${pkgStream.description}` : ''}`;
              commentVariablesInYaml(streamItem, (_pkgStream$vars = pkgStream.vars) !== null && _pkgStream$vars !== void 0 ? _pkgStream$vars : []);
            }
          }
        });
      }
    });
  }
  return doc.toString();
}
function commentVariablesInYaml(rootNode, vars = []) {
  // Node need to be deleted after the end of the visit to be able to visit every node
  const toDeleteFn = [];
  _yaml.default.visit(rootNode, {
    Scalar(key, node, path) {
      if (node.value) {
        const val = node.value.toString();
        for (const varDef of vars) {
          const placeholder = getPlaceholder(varDef);
          if (val.includes(placeholder)) {
            node.comment = ` ${varDef.title}${varDef.description ? `: ${varDef.description}` : ''}`;
            const paths = [...path].reverse();
            let prevPart = node;
            for (const pathPart of paths) {
              if (_yaml.default.isCollection(pathPart)) {
                // If only one items in the collection comment the whole collection
                if (pathPart.items.length === 1) {
                  continue;
                }
              }
              if (_yaml.default.isSeq(pathPart)) {
                const commentDoc = new _yaml.default.Document(new _yaml.default.YAMLSeq());
                commentDoc.add(prevPart);
                const commentStr = commentDoc.toString().trimEnd();
                pathPart.comment = pathPart.comment ? `${pathPart.comment} ${commentStr}` : ` ${commentStr}`;
                const keyToDelete = prevPart;
                toDeleteFn.push(() => {
                  pathPart.items.forEach((item, index) => {
                    if (item === keyToDelete) {
                      pathPart.delete(new _yaml.default.Scalar(index));
                    }
                  });
                });
                return;
              }
              if (_yaml.default.isMap(pathPart)) {
                if (_yaml.default.isPair(prevPart)) {
                  const commentDoc = new _yaml.default.Document(new _yaml.default.YAMLMap());
                  commentDoc.add(prevPart);
                  const commentStr = commentDoc.toString().trimEnd();
                  pathPart.comment = pathPart.comment ? `${pathPart.comment}\n ${commentStr.toString()}` : ` ${commentStr.toString()}`;
                  const keyToDelete = prevPart.key;
                  toDeleteFn.push(() => pathPart.delete(keyToDelete));
                }
                return;
              }
              prevPart = pathPart;
            }
          }
        }
      }
    }
  });
  toDeleteFn.forEach(deleteFn => deleteFn());
}