"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PolicyWatcher = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _pRetry = _interopRequireDefault(require("p-retry"));
var _agent_policy_config = require("../../common/services/agent_policy_config");
var _agent_policy = require("./agent_policy");
var _app_context = require("./app_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class PolicyWatcher {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "subscription", void 0);
    this.logger = logger;
  }
  start(licenseService) {
    var _licenseService$getLi;
    this.subscription = (_licenseService$getLi = licenseService.getLicenseInformation$()) === null || _licenseService$getLi === void 0 ? void 0 : _licenseService$getLi.subscribe(async license => {
      await (0, _pRetry.default)(this.watch.bind(this, license), {
        retries: 3,
        minTimeout: 1000,
        maxTimeout: 5000,
        onFailedAttempt: error => {
          this.logger.warn(`Failed to process agent policy license compliance (attempt ${error.attemptNumber}/${error.retriesLeft + error.attemptNumber}): ${error.message}`);
        }
      });
    });
  }
  stop() {
    if (this.subscription) {
      this.subscription.unsubscribe();
    }
  }
  async watch(license) {
    const log = this.logger.get('endpoint', 'agentPolicyLicenseWatch');
    const soClient = _app_context.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const agentPolicyFetcher = await _agent_policy.agentPolicyService.fetchAllAgentPolicies(soClient, {
      fields: ['is_protected', 'id', 'revision'],
      spaceId: '*'
    } // Don't forget to extend this to include all fields that are used in the `isAgentPolicyValidForLicense` function
    );
    log.info('Checking agent policies for compliance with the current license.');
    const updatedAgentPolicies = [];
    for await (const agentPolicyPageResults of agentPolicyFetcher) {
      const policiesToUpdate = agentPolicyPageResults.reduce((acc, policy) => {
        if (!(0, _agent_policy_config.isAgentPolicyValidForLicense)(policy, license)) {
          acc.push((0, _agent_policy_config.unsetAgentPolicyAccordingToLicenseLevel)(policy, license));
        }
        return acc;
      }, []);
      if (policiesToUpdate.length === 0) {
        break;
      }
      const savedObjectType = await (0, _agent_policy.getAgentPolicySavedObjectType)();
      const {
        saved_objects: bulkUpdateSavedObjects
      } = await soClient.bulkUpdate(policiesToUpdate.map(policy => {
        var _policyContent$space_;
        const {
          id,
          revision,
          ...policyContent
        } = policy;
        const updatedPolicy = {
          type: savedObjectType,
          id,
          attributes: {
            ...policyContent,
            revision: revision + 1,
            updated_at: new Date().toISOString(),
            updated_by: 'system'
          },
          ...((_policyContent$space_ = policyContent.space_ids) !== null && _policyContent$space_ !== void 0 && _policyContent$space_.length ? {
            namespace: policyContent.space_ids[0]
          } : {})
        };
        return updatedPolicy;
      }));
      updatedAgentPolicies.push(...bulkUpdateSavedObjects);
    }
    const failedPolicies = [];
    updatedAgentPolicies.forEach(policy => {
      if (policy.error) {
        failedPolicies.push({
          id: policy.id,
          error: policy.error
        });
      }
    });
    const updatedPoliciesSuccess = updatedAgentPolicies.filter(policy => !policy.error);
    if (!updatedPoliciesSuccess.length && !failedPolicies.length) {
      log.info(`All agent policies are compliant, nothing to do!`);
    } else if (updatedPoliciesSuccess.length && failedPolicies.length) {
      const totalPolicies = updatedPoliciesSuccess.length + failedPolicies.length;
      const message = `Done - ${failedPolicies.length} out of ${totalPolicies} were unsuccessful. Errors encountered:\n${failedPolicies.map(e => `Policy [${e.id}] failed to update due to error: ${e.error.message}`).join('\n')}`;
      log.error(message);
      throw new Error(message);
    } else if (updatedPoliciesSuccess.length) {
      log.info(`Done - ${updatedPoliciesSuccess.length} out of ${updatedPoliciesSuccess.length} were successful. No errors encountered.`);
    } else {
      const message = `Done - all ${failedPolicies.length} failed to update. Errors encountered:\n${failedPolicies.map(e => `Policy [${e.id}] failed to update due to error: ${e.error.message}`).join('\n')}`;
      log.error(message);
      throw new Error(message);
    }
  }
}
exports.PolicyWatcher = PolicyWatcher;