"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.summarizeCommand = exports.summarize = exports.list = exports.byIndex = void 0;
var _walker = require("../../../walker");
var _visitor = require("../../../visitor");
var _pretty_print = require("../../../pretty_print");
var _builder = require("../../../builder");
var _utils = require("../../../visitor/utils");
var generic = _interopRequireWildcard(require("../../generic"));
var _is = require("../../../ast/is");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Lists all "LIMIT" commands in the query AST.
 *
 * @param ast The root AST node to search for "LIMIT" commands.
 * @returns A collection of "LIMIT" commands.
 */
const list = ast => {
  return generic.commands.list(ast, cmd => cmd.name === 'stats');
};

/**
 * Retrieves the "LIMIT" command at the specified index in order of appearance.
 *
 * @param ast The root AST node to search for "LIMIT" commands.
 * @param index The index of the "LIMIT" command to retrieve.
 * @returns The "LIMIT" command at the specified index, if any.
 */
exports.list = list;
const byIndex = (ast, index) => {
  return [...list(ast)][index];
};

/**
 * Summary of a STATS command.
 */

/**
 * Summary of STATS command "aggregates" section (main arguments).
 *
 *    STATS <aggregates> [ BY <grouping> ]
 */
exports.byIndex = byIndex;
const summarizeArgParts = (query, arg) => {
  if ((0, _is.isParamLiteral)(arg)) {
    return [_pretty_print.LeafPrinter.param(arg), arg, arg];
  }
  if ((0, _is.isColumn)(arg)) {
    return [_pretty_print.LeafPrinter.column(arg), arg, arg];
  }
  if ((0, _is.isFunctionExpression)(arg) && arg.name === '=' && (0, _is.isColumn)(arg.args[0])) {
    const [column, definition] = (0, _utils.singleItems)(arg.args);
    return [_pretty_print.LeafPrinter.column(column), column, definition];
  }
  const name = [...query.src].slice(arg.location.min, arg.location.max + 1).join('');
  const args = [_builder.Builder.identifier({
    name
  })];
  const column = _builder.Builder.expression.column({
    args
  });
  return [_pretty_print.LeafPrinter.column(column), column, arg];
};
const summarizeField = (query, arg) => {
  const [field, column, definition] = summarizeArgParts(query, arg);
  const terminals = [];
  const usedFields = new Set();
  _walker.Walker.walk(definition, {
    visitLiteral(node) {
      terminals.push(node);
    },
    visitColumn(node) {
      terminals.push(node);
      usedFields.add(_pretty_print.LeafPrinter.column(node));
    },
    visitListLiteral(node) {
      terminals.push(node);
    },
    visitTimeIntervalLiteral(node) {
      terminals.push(node);
    }
  });
  const summary = {
    arg,
    field,
    column,
    definition,
    terminals,
    usedFields
  };
  return summary;
};

/**
 * Returns a summary of the STATS command.
 *
 * @param query Query which contains the AST and source code.
 * @param command The STATS command AST node to summarize.
 * @returns Summary of the STATS command.
 */
const summarizeCommand = (query, command) => {
  const aggregates = {};
  const grouping = {};
  const newFields = new Set();
  const usedFields = new Set();

  // Process main arguments, the "aggregates" part of the command.
  new _visitor.Visitor().on('visitExpression', ctx => {
    const summary = summarizeField(query, ctx.node);
    aggregates[summary.field] = summary;
    newFields.add(summary.field);
    for (const field of summary.usedFields) usedFields.add(field);
  }).on('visitCommand', () => {}).on('visitStatsCommand', ctx => {
    for (const _ of ctx.visitArguments());
  }).visitCommand(command);

  // Process the "BY" arguments, the "grouping" part of the command.
  new _visitor.Visitor().on('visitExpression', ctx => {
    const node = ctx.node;
    const summary = summarizeField(query, node);
    if ((0, _is.isFunctionExpression)(summary.arg)) {
      // only mark as new field if the arg is a function expression
      newFields.add(summary.field);
    }
    for (const field of summary.usedFields) usedFields.add(field);
    grouping[summary.field] = summary;
  }).on('visitCommandOption', ctx => {
    if (ctx.node.name !== 'by') return;
    for (const _ of ctx.visitArguments());
  }).on('visitCommand', () => {}).on('visitStatsCommand', ctx => {
    for (const _ of ctx.visitOptions());
  }).visitCommand(command);
  const summary = {
    command,
    aggregates,
    grouping,
    newFields,
    usedFields
  };
  return summary;
};

/**
 * Summarizes all STATS commands in the query.
 *
 * @param query Query to summarize.
 * @returns Returns a list of summaries for all STATS commands in the query in
 *     order of appearance.
 */
exports.summarizeCommand = summarizeCommand;
const summarize = query => {
  const summaries = [];
  for (const command of list(query.ast)) {
    const summary = summarizeCommand(query, command);
    summaries.push(summary);
  }
  return summaries;
};
exports.summarize = summarize;