"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDiscoverAppStateContainer = exports.DiscoverAppStateProvider = void 0;
exports.getInitialState = getInitialState;
exports.isEqualFilters = isEqualFilters;
exports.isEqualState = isEqualState;
exports.setState = setState;
exports.useAppStateSelector = void 0;
var _common = require("@kbn/kibana-utils-plugin/common");
var _esQuery = require("@kbn/es-query");
var _public = require("@kbn/kibana-utils-plugin/public");
var _lodash = require("lodash");
var _public2 = require("@kbn/data-plugin/public");
var _add_log = require("../../../utils/add_log");
var _cleanup_url_state = require("./utils/cleanup_url_state");
var _get_state_defaults = require("./utils/get_state_defaults");
var _state_helpers = require("../../../utils/state_helpers");
var _data_sources = require("../../../../common/data_sources");
var _redux = require("./redux");
var _common2 = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const {
  Provider: DiscoverAppStateProvider,
  useSelector: useAppStateSelector
} = (0, _common.createStateContainerReactHelpers)();

/**
 * This is the app state container for Discover main, it's responsible for syncing state with the URL
 * @param stateStorage
 * @param savedSearch
 * @param services
 */
exports.useAppStateSelector = useAppStateSelector;
exports.DiscoverAppStateProvider = DiscoverAppStateProvider;
const getDiscoverAppStateContainer = ({
  tabId,
  stateStorage,
  internalState,
  savedSearchContainer,
  services,
  injectCurrentTab
}) => {
  let initialState = getInitialState({
    initialUrlState: getCurrentUrlState(stateStorage, services),
    savedSearch: savedSearchContainer.getState(),
    services
  });
  let previousState = initialState;
  const appStateContainer = (0, _common.createStateContainer)(initialState);
  const enhancedAppContainer = {
    ...appStateContainer,
    set: value => {
      if (!value) {
        return;
      }
      previousState = appStateContainer.getState();

      // When updating to an ES|QL query, sync the data source
      if ((0, _esQuery.isOfAggregateQueryType)(value.query)) {
        value = {
          ...value,
          dataSource: (0, _data_sources.createEsqlDataSource)()
        };
      }
      appStateContainer.set(value);
    }
  };
  const hasChanged = () => {
    return !isEqualState(initialState, enhancedAppContainer.getState());
  };
  const getAppStateFromSavedSearch = newSavedSearch => {
    return getInitialState({
      initialUrlState: undefined,
      savedSearch: newSavedSearch,
      services
    });
  };
  const resetToState = state => {
    (0, _add_log.addLog)('[appState] reset state to', state);
    previousState = state;
    enhancedAppContainer.set(state);
  };
  const resetInitialState = () => {
    (0, _add_log.addLog)('[appState] reset initial state to the current state');
    initialState = enhancedAppContainer.getState();
  };
  const replaceUrlState = async (newPartial = {}, merge = true) => {
    (0, _add_log.addLog)('[appState] replaceUrlState', {
      newPartial,
      merge
    });
    const state = merge ? {
      ...enhancedAppContainer.getState(),
      ...newPartial
    } : newPartial;
    if (internalState.getState().tabs.unsafeCurrentId === tabId) {
      await stateStorage.set(_common2.APP_STATE_URL_KEY, state, {
        replace: true
      });
    } else {
      enhancedAppContainer.set(state);
    }
  };
  const startAppStateUrlSync = () => {
    (0, _add_log.addLog)('[appState] start syncing state with URL');
    return (0, _public.syncState)({
      storageKey: _common2.APP_STATE_URL_KEY,
      stateContainer: enhancedAppContainer,
      stateStorage
    });
  };
  const initializeAndSync = () => {
    const currentSavedSearch = savedSearchContainer.getState();
    (0, _add_log.addLog)('[appState] initialize state and sync with URL', currentSavedSearch);

    // Set the default profile state only if not loading a saved search,
    // to avoid overwriting saved search state
    if (!currentSavedSearch.id) {
      const {
        breakdownField,
        columns,
        rowHeight,
        hideChart
      } = getCurrentUrlState(stateStorage, services);

      // Only set default state which is not already set in the URL
      internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setResetDefaultProfileState)({
        resetDefaultProfileState: {
          columns: columns === undefined,
          rowHeight: rowHeight === undefined,
          breakdownField: breakdownField === undefined,
          hideChart: hideChart === undefined
        }
      }));
    }
    const {
      data
    } = services;
    const savedSearchDataView = currentSavedSearch.searchSource.getField('index');
    const appState = enhancedAppContainer.getState();
    const setDataViewFromSavedSearch = !appState.dataSource || (0, _data_sources.isDataSourceType)(appState.dataSource, _data_sources.DataSourceType.DataView) && appState.dataSource.dataViewId !== (savedSearchDataView === null || savedSearchDataView === void 0 ? void 0 : savedSearchDataView.id);
    if (setDataViewFromSavedSearch) {
      // used data view is different from the given by url/state which is invalid
      setState(enhancedAppContainer, {
        dataSource: savedSearchDataView !== null && savedSearchDataView !== void 0 && savedSearchDataView.id ? (0, _data_sources.createDataViewDataSource)({
          dataViewId: savedSearchDataView.id
        }) : undefined
      });
    }

    // syncs `_a` portion of url with query services
    const stopSyncingQueryAppStateWithStateContainer = (0, _public2.connectToQueryState)(data.query, enhancedAppContainer, {
      filters: _esQuery.FilterStateStore.APP_STATE,
      query: true
    });

    // syncs `_g` portion of url with query services
    const {
      stop: stopSyncingGlobalStateWithUrl
    } = (0, _public2.syncGlobalQueryStateWithUrl)(data.query, stateStorage);
    const {
      start,
      stop
    } = startAppStateUrlSync();

    // current state need to be pushed to url
    replaceUrlState({}).then(() => start());
    return () => {
      stopSyncingQueryAppStateWithStateContainer();
      stopSyncingGlobalStateWithUrl();
      stop();
    };
  };
  const update = (newPartial, replace = false) => {
    (0, _add_log.addLog)('[appState] update', {
      newPartial,
      replace
    });
    if (replace) {
      return replaceUrlState(newPartial);
    } else {
      previousState = {
        ...enhancedAppContainer.getState()
      };
      setState(enhancedAppContainer, newPartial);
    }
  };
  const getPrevious = () => previousState;
  return {
    ...enhancedAppContainer,
    getPrevious,
    hasChanged,
    initAndSync: initializeAndSync,
    resetToState,
    resetInitialState,
    replaceUrlState,
    syncState: startAppStateUrlSync,
    update,
    getAppStateFromSavedSearch
  };
};
exports.getDiscoverAppStateContainer = getDiscoverAppStateContainer;
function getCurrentUrlState(stateStorage, services) {
  var _cleanupUrlState, _stateStorage$get;
  return (_cleanupUrlState = (0, _cleanup_url_state.cleanupUrlState)((_stateStorage$get = stateStorage.get(_common2.APP_STATE_URL_KEY)) !== null && _stateStorage$get !== void 0 ? _stateStorage$get : {}, services.uiSettings)) !== null && _cleanupUrlState !== void 0 ? _cleanupUrlState : {};
}
function getInitialState({
  initialUrlState,
  savedSearch,
  overrideDataView,
  services
}) {
  const defaultAppState = (0, _get_state_defaults.getStateDefaults)({
    savedSearch,
    overrideDataView,
    services
  });
  const mergedState = {
    ...defaultAppState,
    ...initialUrlState
  };

  // https://github.com/elastic/kibana/issues/122555
  if (typeof mergedState.hideChart !== 'boolean') {
    mergedState.hideChart = undefined;
  }

  // Don't allow URL state to overwrite the data source if there's an ES|QL query
  if ((0, _esQuery.isOfAggregateQueryType)(mergedState.query) && !(0, _data_sources.isEsqlSource)(mergedState.dataSource)) {
    mergedState.dataSource = (0, _data_sources.createEsqlDataSource)();
  }
  return (0, _state_helpers.handleSourceColumnState)(mergedState, services.uiSettings);
}

/**
 * Helper function to merge a given new state with the existing state and to set the given state
 * container
 */
function setState(stateContainer, newState) {
  (0, _add_log.addLog)('[appstate] setState', {
    newState
  });
  const oldState = stateContainer.getState();
  const mergedState = {
    ...oldState,
    ...newState
  };
  if (!isEqualState(oldState, mergedState)) {
    stateContainer.set(mergedState);
  }
}

/**
 * Helper function to compare 2 different filter states
 */
function isEqualFilters(filtersA, filtersB, comparatorOptions = _esQuery.COMPARE_ALL_OPTIONS) {
  if (!filtersA && !filtersB) {
    return true;
  } else if (!filtersA || !filtersB) {
    return false;
  }
  return (0, _esQuery.compareFilters)(filtersA, filtersB, comparatorOptions);
}

/**
 * Helper function to compare 2 different state, is needed since comparing filters
 * works differently
 */
function isEqualState(stateA, stateB, exclude = []) {
  if (!stateA && !stateB) {
    return true;
  } else if (!stateA || !stateB) {
    return false;
  }
  const {
    filters: stateAFilters = [],
    ...stateAPartial
  } = (0, _lodash.omit)(stateA, exclude);
  const {
    filters: stateBFilters = [],
    ...stateBPartial
  } = (0, _lodash.omit)(stateB, exclude);
  return (0, _lodash.isEqual)(stateAPartial, stateBPartial) && isEqualFilters(stateAFilters, stateBFilters);
}