"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.internalStateSlice = exports.defaultTabState = exports.createInternalStateStore = void 0;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _toolkit = require("@reduxjs/toolkit");
var _runtime_state = require("./runtime_state");
var _types = require("./types");
var _data_views = require("./actions/data_views");
var _selectors = require("./selectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const MIDDLEWARE_THROTTLE_MS = 300;
const MIDDLEWARE_THROTTLE_OPTIONS = {
  leading: false,
  trailing: true
};
const defaultTabState = exports.defaultTabState = {
  lastPersistedGlobalState: {},
  dataViewId: undefined,
  isDataViewLoading: false,
  dataRequestParams: {
    timeRangeAbsolute: undefined,
    timeRangeRelative: undefined,
    searchSessionId: undefined
  },
  overriddenVisContextAfterInvalidation: undefined,
  resetDefaultProfileState: {
    resetId: '',
    columns: false,
    rowHeight: false,
    breakdownField: false,
    hideChart: false
  },
  documentsRequest: {
    loadingStatus: _types.LoadingStatus.Uninitialized,
    result: []
  },
  totalHitsRequest: {
    loadingStatus: _types.LoadingStatus.Uninitialized,
    result: 0
  },
  chartRequest: {
    loadingStatus: _types.LoadingStatus.Uninitialized,
    result: {}
  },
  uiState: {}
};
const initialState = {
  initializationState: {
    hasESData: false,
    hasUserDataView: false
  },
  defaultProfileAdHocDataViewIds: [],
  savedDataViews: [],
  expandedDoc: undefined,
  isESQLToDataViewTransitionModalVisible: false,
  tabs: {
    byId: {},
    allIds: [],
    unsafeCurrentId: '',
    recentlyClosedTabIds: []
  }
};
const withTab = (state, action, fn) => {
  const tab = (0, _selectors.selectTab)(state, action.payload.tabId);
  if (tab) {
    fn(tab);
  }
};
const internalStateSlice = exports.internalStateSlice = (0, _toolkit.createSlice)({
  name: 'internalState',
  initialState,
  reducers: {
    setInitializationState: (state, action) => {
      state.initializationState = action.payload;
    },
    setTabs: (state, action) => {
      state.tabs.byId = [...action.payload.recentlyClosedTabs, ...action.payload.allTabs].reduce((acc, tab) => ({
        ...acc,
        [tab.id]: tab
      }), {});
      state.tabs.allIds = action.payload.allTabs.map(tab => tab.id);
      state.tabs.unsafeCurrentId = action.payload.selectedTabId;
      state.tabs.recentlyClosedTabIds = action.payload.recentlyClosedTabs.map(tab => tab.id);
    },
    setDataViewId: (state, action) => withTab(state, action, tab => {
      if (action.payload.dataViewId !== tab.dataViewId) {
        state.expandedDoc = undefined;
      }
      tab.dataViewId = action.payload.dataViewId;
    }),
    setIsDataViewLoading: (state, action) => withTab(state, action, tab => {
      tab.isDataViewLoading = action.payload.isDataViewLoading;
    }),
    setDefaultProfileAdHocDataViewIds: (state, action) => {
      state.defaultProfileAdHocDataViewIds = action.payload;
    },
    setExpandedDoc: (state, action) => {
      state.expandedDoc = action.payload.expandedDoc;
      state.initialDocViewerTabId = action.payload.initialDocViewerTabId;
    },
    setDataRequestParams: (state, action) => withTab(state, action, tab => {
      tab.dataRequestParams = action.payload.dataRequestParams;
    }),
    setTabAppStateAndGlobalState: (state, action) => withTab(state, action, tab => {
      tab.lastPersistedGlobalState = action.payload.globalState || {};
    }),
    setOverriddenVisContextAfterInvalidation: (state, action) => withTab(state, action, tab => {
      tab.overriddenVisContextAfterInvalidation = action.payload.overriddenVisContextAfterInvalidation;
    }),
    setIsESQLToDataViewTransitionModalVisible: (state, action) => {
      state.isESQLToDataViewTransitionModalVisible = action.payload;
    },
    setResetDefaultProfileState: {
      prepare: payload => ({
        payload: {
          ...payload,
          resetDefaultProfileState: {
            ...payload.resetDefaultProfileState,
            resetId: (0, _uuid.v4)()
          }
        }
      }),
      reducer: (state, action) => withTab(state, action, tab => {
        tab.resetDefaultProfileState = action.payload.resetDefaultProfileState;
      })
    },
    resetOnSavedSearchChange: (state, action) => withTab(state, action, tab => {
      tab.overriddenVisContextAfterInvalidation = undefined;
      state.expandedDoc = undefined;
    }),
    setDataGridUiState: (state, action) => withTab(state, action, tab => {
      tab.uiState.dataGrid = action.payload.dataGridUiState;
    }),
    setFieldListUiState: (state, action) => withTab(state, action, tab => {
      tab.uiState.fieldList = action.payload.fieldListUiState;
    })
  },
  extraReducers: builder => {
    builder.addCase(_data_views.loadDataViewList.fulfilled, (state, action) => {
      state.savedDataViews = action.payload;
    });
  }
});
const createMiddleware = ({
  tabsStorageManager,
  runtimeStateManager
}) => {
  const listenerMiddleware = (0, _toolkit.createListenerMiddleware)();
  listenerMiddleware.startListening({
    actionCreator: internalStateSlice.actions.setTabs,
    effect: (0, _lodash.throttle)(action => {
      const getTabAppState = tabId => (0, _runtime_state.selectTabRuntimeAppState)(runtimeStateManager, tabId);
      void tabsStorageManager.persistLocally(action.payload, getTabAppState);
    }, MIDDLEWARE_THROTTLE_MS, MIDDLEWARE_THROTTLE_OPTIONS)
  });
  listenerMiddleware.startListening({
    actionCreator: internalStateSlice.actions.setTabAppStateAndGlobalState,
    effect: (0, _lodash.throttle)(action => {
      tabsStorageManager.updateTabStateLocally(action.payload.tabId, action.payload);
    }, MIDDLEWARE_THROTTLE_MS, MIDDLEWARE_THROTTLE_OPTIONS)
  });
  return listenerMiddleware;
};
const IS_JEST_ENVIRONMENT = typeof jest !== 'undefined';
const createInternalStateStore = options => {
  return (0, _toolkit.configureStore)({
    reducer: internalStateSlice.reducer,
    middleware: getDefaultMiddleware => getDefaultMiddleware({
      thunk: {
        extraArgument: options
      },
      serializableCheck: !IS_JEST_ENVIRONMENT
    }).prepend(createMiddleware(options).middleware),
    devTools: {
      name: 'DiscoverInternalState'
    }
  });
};
exports.createInternalStateStore = createInternalStateStore;