"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useRedirectLink = void 0;
var _common = require("@kbn/discover-plugin/common");
var _esQuery = require("@kbn/es-query");
var _routerUtils = require("@kbn/router-utils");
var _react = require("react");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useRedirectLink = ({
  dataStreamStat,
  query,
  timeRangeConfig,
  breakdownField,
  sendTelemetry,
  selector
}) => {
  const {
    services: {
      share
    }
  } = (0, _utils.useKibanaContextForPlugin)();
  const {
    from,
    to
  } = timeRangeConfig;
  return (0, _react.useMemo)(() => {
    const config = buildDiscoverConfig({
      locatorClient: share.url.locators,
      dataStreamStat,
      query,
      from,
      to,
      breakdownField,
      selector
    });
    const onClickWithTelemetry = event => {
      sendTelemetry();
      if (config.routerLinkProps.onClick) {
        config.routerLinkProps.onClick(event);
      }
    };
    const navigateWithTelemetry = () => {
      sendTelemetry();
      config.navigate();
    };
    return {
      linkProps: {
        ...config.routerLinkProps,
        onClick: onClickWithTelemetry
      },
      navigate: navigateWithTelemetry,
      isLogsExplorerAvailable: false
    };
  }, [share.url.locators, dataStreamStat, query, from, to, breakdownField, selector, sendTelemetry]);
};
exports.useRedirectLink = useRedirectLink;
const buildDiscoverConfig = ({
  locatorClient,
  dataStreamStat,
  query,
  from,
  to,
  breakdownField,
  selector
}) => {
  const {
    dataViewId,
    dataViewTitle
  } = getDataView({
    dataStreamStat,
    selector
  });
  const filters = getFilters({
    dataStreamStat,
    dataViewId,
    dataViewTitle,
    selector
  });
  const params = {
    timeRange: {
      from,
      to
    },
    refreshInterval: {
      pause: true,
      value: 60000
    },
    dataViewId,
    dataViewSpec: {
      id: dataViewId,
      title: dataViewId,
      timeFieldName: '@timestamp'
    },
    query,
    breakdownField,
    columns: [],
    filters,
    interval: 'auto',
    sort: [['@timestamp', 'desc']]
  };
  const locator = locatorClient.get(_common.DISCOVER_APP_LOCATOR);
  const urlToDiscover = locator === null || locator === void 0 ? void 0 : locator.getRedirectUrl(params);
  const navigateToDiscover = () => {
    locator === null || locator === void 0 ? void 0 : locator.navigate(params);
  };
  const discoverLinkProps = (0, _routerUtils.getRouterLinkProps)({
    href: urlToDiscover,
    onClick: navigateToDiscover
  });
  return {
    routerLinkProps: discoverLinkProps,
    navigate: navigateToDiscover
  };
};
const getDataView = ({
  dataStreamStat,
  selector
}) => {
  if (dataStreamStat && typeof dataStreamStat === 'string') {
    return {
      dataViewId: dataStreamStat,
      dataViewTitle: dataStreamStat
    };
  }
  const {
    name,
    namespace,
    type,
    integration
  } = dataStreamStat;
  const dataViewNamespace = `${namespace || '*'}`;
  const dataViewSelector = selector ? `${selector}` : '';
  const dataViewId = `${type}-${name}-${dataViewNamespace}${dataViewSelector}`;
  const dataViewTitle = integration ? `[${integration.title}] ${name}-${dataViewNamespace}${dataViewSelector}` : `${dataViewId}`;
  return {
    dataViewId,
    dataViewTitle
  };
};
const getFilters = ({
  dataStreamStat,
  dataViewId,
  dataViewTitle,
  selector
}) => {
  if (dataStreamStat && typeof dataStreamStat === 'string') {
    return [];
  }
  return selector ? [] : [(0, _esQuery.buildPhraseFilter)({
    name: 'data_stream.namespace',
    type: 'string'
  }, dataStreamStat.namespace, {
    id: dataViewId,
    title: dataViewTitle
  })];
};