"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineBulkCreateOrUpdateRolesRoutes = defineBulkCreateOrUpdateRolesRoutes;
var _coreSecurityServer = require("@kbn/core-security-server");
var _lib = require("./lib");
var _model = require("./model");
var _constants = require("../../../../common/constants");
var _errors = require("../../../errors");
var _lib2 = require("../../../lib");
var _licensed_route_handler = require("../../licensed_route_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function defineBulkCreateOrUpdateRolesRoutes({
  router,
  authz,
  getFeatures,
  getFeatureUsageService
}) {
  router.versioned.post({
    path: '/api/security/roles',
    access: 'public',
    summary: 'Create or update roles',
    options: {
      tags: ['oas-tag:roles']
    },
    security: {
      authz: _coreSecurityServer.AuthzDisabled.delegateToESClient
    }
  }).addVersion({
    version: _constants.API_VERSIONS.roles.public.v1,
    validate: {
      request: {
        body: (0, _model.getBulkCreateOrUpdatePayloadSchema)(() => {
          const privileges = authz.privileges.get();
          return {
            global: Object.keys(privileges.global),
            space: Object.keys(privileges.space)
          };
        })
      },
      response: {
        200: {
          description: 'Indicates a successful call.'
        }
      }
    }
  }, (0, _licensed_route_handler.createLicensedRouteHandler)(async (context, request, response) => {
    try {
      var _esErrors$details;
      const esClient = (await context.core).elasticsearch.client;
      const features = await getFeatures();
      const {
        roles
      } = request.body;
      const validatedRolesNames = [];
      const kibanaErrors = {};
      for (const [roleName, role] of Object.entries(roles)) {
        const {
          validationErrors
        } = (0, _lib2.validateKibanaPrivileges)(features, role.kibana);
        if (validationErrors.length) {
          kibanaErrors[roleName] = {
            type: 'kibana_privilege_validation_exception',
            reason: `Role cannot be updated due to validation errors: ${JSON.stringify(validationErrors)}`
          };
          continue;
        }
        validatedRolesNames.push(roleName);
      }
      const rawRoles = await esClient.asCurrentUser.security.getRole({
        name: validatedRolesNames.join(',')
      }, {
        ignore: [404]
      });
      const esRolesPayload = Object.fromEntries(validatedRolesNames.map(roleName => [roleName, (0, _model.transformPutPayloadToElasticsearchRole)(roles[roleName], authz.applicationName, rawRoles[roleName] ? rawRoles[roleName].applications : [])]));
      const esResponse = await esClient.asCurrentUser.transport.request({
        method: 'POST',
        path: '/_security/role',
        body: {
          roles: esRolesPayload
        }
      });
      for (const roleName of [...((_esResponse$created = esResponse.created) !== null && _esResponse$created !== void 0 ? _esResponse$created : []), ...((_esResponse$updated = esResponse.updated) !== null && _esResponse$updated !== void 0 ? _esResponse$updated : []), ...((_esResponse$noop = esResponse.noop) !== null && _esResponse$noop !== void 0 ? _esResponse$noop : [])]) {
        var _esResponse$created, _esResponse$updated, _esResponse$noop;
        if ((0, _lib.roleGrantsSubFeaturePrivileges)(features, roles[roleName])) {
          getFeatureUsageService().recordSubFeaturePrivilegeUsage();
        }
      }
      const {
        created,
        noop,
        updated,
        errors: esErrors
      } = esResponse;
      const hasAnyErrors = Object.keys(kibanaErrors).length || (esErrors === null || esErrors === void 0 ? void 0 : esErrors.count);
      return response.ok({
        body: {
          created,
          noop,
          updated,
          ...(hasAnyErrors && {
            errors: {
              ...kibanaErrors,
              ...((_esErrors$details = esErrors === null || esErrors === void 0 ? void 0 : esErrors.details) !== null && _esErrors$details !== void 0 ? _esErrors$details : {})
            }
          })
        }
      });
    } catch (error) {
      return response.customError((0, _errors.wrapIntoCustomErrorResponse)(error));
    }
  }));
}