"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSavedQueryRoute = void 0;
var _lodash = require("lodash");
var _get_internal_saved_object_client = require("../../utils/get_internal_saved_object_client");
var _constants = require("../../../common/constants");
var _common = require("../../../common");
var _types = require("../../../common/types");
var _route_validation = require("../../utils/build_validation/route_validation");
var _utils = require("../utils");
var _api = require("../../../common/api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createSavedQueryRoute = (router, osqueryContext) => {
  router.versioned.post({
    access: 'public',
    path: '/api/osquery/saved_queries',
    security: {
      authz: {
        requiredPrivileges: [`${_common.PLUGIN_ID}-writeSavedQueries`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidation)(_api.createSavedQueryRequestSchema)
      }
    }
  }, async (context, request, response) => {
    var _coreContext$security;
    const coreContext = await context.core;
    const spaceScopedClient = await (0, _get_internal_saved_object_client.createInternalSavedObjectsClientForSpaceId)(osqueryContext, request);
    const {
      id,
      description,
      platform,
      query,
      version,
      interval,
      snapshot,
      removed,
      timeout,
      // eslint-disable-next-line @typescript-eslint/naming-convention
      ecs_mapping
    } = request.body;
    const currentUser = (_coreContext$security = coreContext.security.authc.getCurrentUser()) === null || _coreContext$security === void 0 ? void 0 : _coreContext$security.username;
    const conflictingEntries = await spaceScopedClient.find({
      type: _types.savedQuerySavedObjectType,
      filter: `${_types.savedQuerySavedObjectType}.attributes.id: "${id}"`
    });
    if (conflictingEntries.saved_objects.length && (0, _lodash.some)(conflictingEntries.saved_objects, ['attributes.id', id])) {
      return response.conflict({
        body: `Saved query with id "${id}" already exists.`
      });
    }
    const savedQuerySO = await spaceScopedClient.create(_types.savedQuerySavedObjectType, (0, _lodash.pickBy)({
      id,
      description,
      query,
      platform,
      version,
      interval,
      snapshot,
      removed,
      timeout,
      ecs_mapping: (0, _utils.convertECSMappingToArray)(ecs_mapping),
      created_by: currentUser,
      created_at: new Date().toISOString(),
      updated_by: currentUser,
      updated_at: new Date().toISOString()
    }, value => !(0, _lodash.isEmpty)(value) || (0, _lodash.isBoolean)(value) || (0, _lodash.isNumber)(value)));
    const {
      attributes
    } = savedQuerySO;
    const data = (0, _lodash.pickBy)({
      created_at: attributes.created_at,
      created_by: attributes.created_by,
      description: attributes.description,
      id: attributes.id,
      removed: attributes.removed,
      snapshot: attributes.snapshot,
      version: attributes.version,
      interval: attributes.interval,
      timeout: attributes.timeout,
      platform: attributes.platform,
      query: attributes.query,
      updated_at: attributes.updated_at,
      updated_by: attributes.updated_by,
      saved_object_id: savedQuerySO.id,
      ecs_mapping
    }, value => !(0, _lodash.isEmpty)(value) || (0, _lodash.isNumber)(value));
    return response.ok({
      body: {
        data
      }
    });
  });
};
exports.createSavedQueryRoute = createSavedQueryRoute;