"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.binaryExpressionGroup = exports.BinaryExpressionGroup = void 0;
var _is = require("./is");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
/**
 * The group name of a binary expression. Groups are ordered by precedence.
 */
let BinaryExpressionGroup = exports.BinaryExpressionGroup = /*#__PURE__*/function (BinaryExpressionGroup) {
  BinaryExpressionGroup[BinaryExpressionGroup["none"] = 0] = "none";
  BinaryExpressionGroup[BinaryExpressionGroup["unknown"] = 1] = "unknown";
  BinaryExpressionGroup[BinaryExpressionGroup["or"] = 10] = "or";
  BinaryExpressionGroup[BinaryExpressionGroup["and"] = 11] = "and";
  BinaryExpressionGroup[BinaryExpressionGroup["regex"] = 20] = "regex";
  BinaryExpressionGroup[BinaryExpressionGroup["assignment"] = 30] = "assignment";
  BinaryExpressionGroup[BinaryExpressionGroup["comparison"] = 40] = "comparison";
  BinaryExpressionGroup[BinaryExpressionGroup["additive"] = 50] = "additive";
  BinaryExpressionGroup[BinaryExpressionGroup["multiplicative"] = 60] = "multiplicative";
  return BinaryExpressionGroup;
}({});
/**
 * Returns the group of a binary expression.
 *
 * @param node Any ES|QL AST node.
 * @returns Binary expression group or undefined if the node is
 *     not a binary expression.
 */
const binaryExpressionGroup = node => {
  if ((0, _is.isBinaryExpression)(node)) {
    switch (node.name) {
      case '+':
      case '-':
        return BinaryExpressionGroup.additive;
      case '*':
      case '/':
      case '%':
        return BinaryExpressionGroup.multiplicative;
      case '=':
        return BinaryExpressionGroup.assignment;
      case '==':
      case '=~':
      case '!=':
      case '<':
      case '<=':
      case '>':
      case '>=':
        return BinaryExpressionGroup.comparison;
      case 'like':
      case 'not like':
      case 'rlike':
      case 'not rlike':
        return BinaryExpressionGroup.regex;
      case 'or':
        return BinaryExpressionGroup.or;
      case 'and':
        return BinaryExpressionGroup.and;
    }
    return BinaryExpressionGroup.unknown;
  }
  return BinaryExpressionGroup.none;
};
exports.binaryExpressionGroup = binaryExpressionGroup;