"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataStreamAdapter = getDataStreamAdapter;
var _create_concrete_write_index = require("./create_concrete_write_index");
var _retry_transient_es_errors = require("./retry_transient_es_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// eslint-disable-next-line max-classes-per-file

function getDataStreamAdapter(opts) {
  if (opts.useDataStreamForAlerts) {
    return new DataStreamImplementation();
  } else {
    return new AliasImplementation();
  }
}

// implementation using data streams
class DataStreamImplementation {
  isUsingDataStreams() {
    return true;
  }
  getIndexTemplateFields(alias, patterns) {
    return {
      data_stream: {
        hidden: true
      },
      index_patterns: [alias]
    };
  }
  async createStream(opts) {
    return createDataStream(opts);
  }
}

// implementation using aliases and backing indices
class AliasImplementation {
  isUsingDataStreams() {
    return false;
  }
  getIndexTemplateFields(alias, patterns) {
    return {
      index_patterns: patterns,
      rollover_alias: alias
    };
  }
  async createStream(opts) {
    return createAliasStream(opts);
  }
}
async function createDataStream(opts) {
  const {
    logger,
    esClient,
    indexPatterns,
    totalFieldsLimit
  } = opts;
  logger.debug(`Creating data stream - ${indexPatterns.alias}`);

  // check if data stream exists
  let dataStreamExists = false;
  try {
    const response = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.getDataStream({
      name: indexPatterns.alias,
      expand_wildcards: 'all'
    }), {
      logger
    });
    dataStreamExists = response.data_streams.length > 0;
  } catch (error) {
    if ((error === null || error === void 0 ? void 0 : error.statusCode) !== 404) {
      logger.error(`Error fetching data stream for ${indexPatterns.alias} - ${error.message}`);
      throw error;
    }
  }

  // if a data stream exists, update the underlying mapping
  if (dataStreamExists) {
    await (0, _create_concrete_write_index.updateIndexMappings)({
      logger,
      esClient,
      totalFieldsLimit,
      concreteIndices: [{
        alias: indexPatterns.alias,
        index: indexPatterns.alias,
        isWriteIndex: true
      }]
    });
  } else {
    try {
      await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.createDataStream({
        name: indexPatterns.alias
      }), {
        logger
      });
    } catch (error) {
      var _error$meta, _error$meta$body, _error$meta$body$erro;
      if ((error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : _error$meta$body$erro.type) !== 'resource_already_exists_exception') {
        logger.error(`Error creating data stream ${indexPatterns.alias} - ${error.message}`);
        throw error;
      }
    }
  }
}
async function createAliasStream(opts) {
  const {
    logger,
    esClient,
    indexPatterns,
    totalFieldsLimit
  } = opts;
  logger.debug(`Creating concrete write index - ${indexPatterns.name}`);

  // check if a concrete write index already exists
  let concreteIndices = [];
  try {
    // Specify both the index pattern for the backing indices and their aliases
    // The alias prevents the request from finding other namespaces that could match the -* pattern
    const patterns = [indexPatterns.pattern];
    if (indexPatterns.reindexedPattern) {
      patterns.push(indexPatterns.reindexedPattern);
    }
    const response = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.getAlias({
      index: patterns,
      name: indexPatterns.basePattern
    }), {
      logger
    });
    concreteIndices = Object.entries(response).flatMap(([index, {
      aliases
    }]) => Object.entries(aliases).map(([aliasName, aliasProperties]) => {
      var _aliasProperties$is_w;
      return {
        index,
        alias: aliasName,
        isWriteIndex: (_aliasProperties$is_w = aliasProperties.is_write_index) !== null && _aliasProperties$is_w !== void 0 ? _aliasProperties$is_w : false
      };
    }));
    logger.info(() => `Found ${concreteIndices.length} concrete indices for ${indexPatterns.name} - ${JSON.stringify(concreteIndices)}`);
  } catch (error) {
    // 404 is expected if no concrete write indices have been created
    if (error.statusCode !== 404) {
      logger.error(`Error fetching concrete indices for ${indexPatterns.pattern} pattern - ${error.message}`);
      throw error;
    }
  }
  let concreteWriteIndicesExist = false;
  // if a concrete write index already exists, update the underlying mapping
  if (concreteIndices.length > 0) {
    await (0, _create_concrete_write_index.updateIndexMappings)({
      logger,
      esClient,
      totalFieldsLimit,
      concreteIndices,
      validIndexPrefixes: indexPatterns.validPrefixes
    });
    const concreteIndicesExist = concreteIndices.some(index => index.alias === indexPatterns.alias);
    concreteWriteIndicesExist = concreteIndices.some(index => index.alias === indexPatterns.alias && index.isWriteIndex);

    // If there are some concrete indices but none of them are the write index, we'll throw an error
    // because one of the existing indices should have been the write target.
    if (concreteIndicesExist && !concreteWriteIndicesExist) {
      logger.debug(`Indices matching pattern ${indexPatterns.pattern} exist but none are set as the write index for alias ${indexPatterns.alias}`);
      await (0, _create_concrete_write_index.setConcreteWriteIndex)({
        logger,
        esClient,
        concreteIndices
      });
      concreteWriteIndicesExist = true;
    }
  }

  // check if a concrete write index already exists
  if (!concreteWriteIndicesExist) {
    try {
      await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.create({
        index: indexPatterns.name,
        aliases: {
          [indexPatterns.alias]: {
            is_write_index: true
          }
        }
      }), {
        logger
      });
    } catch (error) {
      var _error$meta2, _error$meta2$body, _error$meta2$body$err;
      logger.error(`Error creating concrete write index - ${error.message}`);
      // If the index already exists and it's the write index for the alias,
      // something else created it so suppress the error. If it's not the write
      // index, that's bad, throw an error.
      if ((error === null || error === void 0 ? void 0 : (_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$body = _error$meta2.body) === null || _error$meta2$body === void 0 ? void 0 : (_error$meta2$body$err = _error$meta2$body.error) === null || _error$meta2$body$err === void 0 ? void 0 : _error$meta2$body$err.type) === 'resource_already_exists_exception') {
        var _existingIndices$inde, _existingIndices$inde2, _existingIndices$inde3;
        const existingIndices = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.get({
          index: indexPatterns.name
        }), {
          logger
        });
        if (!((_existingIndices$inde = existingIndices[indexPatterns.name]) !== null && _existingIndices$inde !== void 0 && (_existingIndices$inde2 = _existingIndices$inde.aliases) !== null && _existingIndices$inde2 !== void 0 && (_existingIndices$inde3 = _existingIndices$inde2[indexPatterns.alias]) !== null && _existingIndices$inde3 !== void 0 && _existingIndices$inde3.is_write_index)) {
          throw Error(`Attempted to create index: ${indexPatterns.name} as the write index for alias: ${indexPatterns.alias}, but the index already exists and is not the write index for the alias`);
        }
      } else {
        throw error;
      }
    }
  }
}