"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.unknownAgentProviderId = exports.toStructuredAgentIdentifier = exports.toSerializedAgentIdentifier = exports.serializedPartsSeparator = exports.isStructuredAgentIdentifier = exports.isSerializedAgentIdentifier = exports.isPlainAgentIdentifier = void 0;
var _errors = require("../base/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Structured representation of an agent identifier.
 */

/**
 * String representation of {@link StructuredAgentIdentifier}
 * Follow a `{providerId}:{agentId}` format.
 */

const serializedPartsSeparator = exports.serializedPartsSeparator = '::';
const unknownAgentProviderId = exports.unknownAgentProviderId = 'unknown';

/**
 * Check if the given {@link AgentIdentifier} is a {@link SerializedAgentIdentifier}
 */
const isSerializedAgentIdentifier = identifier => {
  return typeof identifier === 'string' && identifier.split(serializedPartsSeparator).length === 2;
};

/**
 * Check if the given {@link AgentIdentifier} is a {@link StructuredAgentIdentifier}
 */
exports.isSerializedAgentIdentifier = isSerializedAgentIdentifier;
const isStructuredAgentIdentifier = identifier => {
  return typeof identifier === 'object' && 'agentId' in identifier && 'providerId' in identifier;
};

/**
 * Check if the given {@link AgentIdentifier} is a {@link PlainIdAgentIdentifier}
 */
exports.isStructuredAgentIdentifier = isStructuredAgentIdentifier;
const isPlainAgentIdentifier = identifier => {
  return typeof identifier === 'string' && identifier.split(serializedPartsSeparator).length === 1;
};

/**
 * Convert the given {@link ToolIdentifier} to a {@link SerializedToolIdentifier}
 */
exports.isPlainAgentIdentifier = isPlainAgentIdentifier;
const toSerializedAgentIdentifier = identifier => {
  if (isSerializedAgentIdentifier(identifier)) {
    return identifier;
  }
  if (isStructuredAgentIdentifier(identifier)) {
    return `${identifier.providerId}::${identifier.agentId}`;
  }
  if (isPlainAgentIdentifier(identifier)) {
    return `${unknownAgentProviderId}::${identifier}`;
  }
  throw (0, _errors.createInternalError)(`Malformed agent identifier: "${identifier}"`);
};

/**
 * Convert the given {@link ToolIdentifier} to a {@link StructuredToolIdentifier}
 */
exports.toSerializedAgentIdentifier = toSerializedAgentIdentifier;
const toStructuredAgentIdentifier = identifier => {
  if (isStructuredAgentIdentifier(identifier)) {
    return identifier;
  }
  if (isSerializedAgentIdentifier(identifier)) {
    const [providerId, agentId] = identifier.split(serializedPartsSeparator);
    return {
      agentId,
      providerId
    };
  }
  if (isPlainAgentIdentifier(identifier)) {
    return {
      agentId: identifier,
      providerId: unknownAgentProviderId
    };
  }
  throw (0, _errors.createInternalError)(`Malformed agent identifier: "${identifier}"`);
};
exports.toStructuredAgentIdentifier = toStructuredAgentIdentifier;