"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.usePrivilegedUsersTableData = void 0;
var _fp = require("lodash/fp");
var _reactQuery = require("@kbn/react-query");
var _esqlUtils = require("@kbn/esql-utils");
var _react = require("react");
var _search_strategy = require("../../../../../../common/search_strategy");
var _esql = require("../../../../../common/utils/esql");
var _use_risk_score = require("../../../../api/hooks/use_risk_score");
var _esql_source_query = require("./esql_source_query");
var _use_global_filter_query = require("../../../../../common/hooks/use_global_filter_query");
var _use_asset_criticality = require("../../../asset_criticality/use_asset_criticality");
var _utils = require("../../../../../../common/entity_analytics/privilege_monitoring/utils");
var _kibana = require("../../../../../common/lib/kibana");
var _ = require(".");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const usePrivilegedUsersTableData = (spaceId, currentPage, toggleStatus) => {
  const {
    data
  } = (0, _kibana.useKibana)().services;
  const privilegedUsersTableQuery = (0, _esql_source_query.getPrivilegedUsersQuery)(spaceId, currentPage * _.DEFAULT_PAGE_SIZE + 1 // we add 1 so that we know if there are more results to show
  );
  const {
    filterQuery: filterQueryWithoutTimerange
  } = (0, _use_global_filter_query.useGlobalFilterQuery)();
  const {
    data: result,
    isLoading: loadingPrivilegedUsers,
    isError: privilegedUsersError,
    refetch
  } = (0, _reactQuery.useQuery)({
    queryKey: ['privileged-users-table', privilegedUsersTableQuery, filterQueryWithoutTimerange],
    enabled: toggleStatus,
    queryFn: async () => {
      return (0, _esqlUtils.getESQLResults)({
        esqlQuery: privilegedUsersTableQuery,
        search: data.search.search,
        filter: filterQueryWithoutTimerange
      });
    }
  });
  const records = (0, _react.useMemo)(() => (0, _esql.esqlResponseToRecords)(result === null || result === void 0 ? void 0 : result.response), [result]);
  const nameFilterQuery = (0, _react.useMemo)(() => {
    const userNames = records.map(user => user['user.name']);
    return (0, _search_strategy.buildEntityNameFilter)(_search_strategy.EntityType.user, userNames);
  }, [records]);
  const {
    data: riskScoreData,
    error: riskScoreError,
    loading: loadingRiskScore,
    hasEngineBeenInstalled: hasRiskEngineBeenInstalled
  } = (0, _use_risk_score.useRiskScore)({
    riskEntity: _search_strategy.EntityType.user,
    filterQuery: nameFilterQuery,
    onlyLatest: true,
    pagination: {
      cursorStart: 0,
      querySize: records.length
    },
    skip: nameFilterQuery === undefined || records.length === 0
  });
  const riskScores = riskScoreData && riskScoreData.length > 0 ? riskScoreData : [];
  const riskScoreByUserName = Object.fromEntries(riskScores.map(riskScore => [riskScore.user.name, riskScore]));
  const {
    data: assetCriticalityData,
    isError: assetCriticalityError,
    isLoading: loadingAssetCriticality
  } = (0, _use_asset_criticality.useAssetCriticalityFetchList)({
    idField: 'user.name',
    idValues: records.map(user => user['user.name']),
    skip: !toggleStatus || records.length === 0
  });
  const assetCriticalityRecords = assetCriticalityData && assetCriticalityData.records.length > 0 ? assetCriticalityData.records : [];
  const assetCriticalityByUserName = Object.fromEntries(assetCriticalityRecords.map(assetCriticalityRecord => [assetCriticalityRecord.id_value, assetCriticalityRecord.criticality_level]));
  const enrichedRecords = (0, _react.useMemo)(() => records.map((record, index) => {
    let enrichedFields = {};
    const riskScore = riskScoreByUserName[record['user.name']];
    if (riskScore) {
      enrichedFields = {
        ...enrichedFields,
        risk_score: riskScore.user.risk.calculated_score_norm,
        risk_level: riskScore.user.risk.calculated_level
      };
    }
    const assetCriticality = assetCriticalityByUserName[record['user.name']];
    if (assetCriticality) {
      enrichedFields = {
        ...enrichedFields,
        criticality_level: assetCriticality
      };
    }
    return {
      ...record,
      ...enrichedFields
    };
  }), [records, riskScoreByUserName, assetCriticalityByUserName]);
  const isLoading = loadingPrivilegedUsers || records.length > 0 && (loadingRiskScore || loadingAssetCriticality);
  const visibleRecords = (0, _fp.take)(currentPage * _.DEFAULT_PAGE_SIZE, enrichedRecords);
  const inspect = (0, _react.useMemo)(() => {
    return {
      dsl: [JSON.stringify({
        index: [(0, _utils.getPrivilegedMonitorUsersIndex)(spaceId)],
        body: (0, _esqlUtils.prettifyQuery)(privilegedUsersTableQuery, false)
      }, null, 2)],
      response: result ? [JSON.stringify(result, null, 2)] : []
    };
  }, [privilegedUsersTableQuery, result, spaceId]);
  return {
    visibleRecords,
    isLoading,
    refetch,
    inspect,
    hasError: privilegedUsersError || hasRiskEngineBeenInstalled && riskScoreError || assetCriticalityError,
    hasNextPage: records.length > currentPage * _.DEFAULT_PAGE_SIZE
  };
};
exports.usePrivilegedUsersTableData = usePrivilegedUsersTableData;