"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.assertRuleUpgradeAfterReview = assertRuleUpgradeAfterReview;
var _react = require("@testing-library/react");
var _lodash = require("lodash");
var _detection_engine = require("../../../../../../../../common/api/detection_engine");
var _rule_upgrade_helpers = require("./rule_upgrade_helpers");
var _set_field_value = require("./set_field_value");
var _rule_upgrade_flyout = require("./rule_upgrade_flyout");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function assertRuleUpgradeAfterReview({
  ruleType,
  fieldName: rawFieldName,
  fieldVersions: {
    initial,
    customized,
    upgrade,
    resolvedValue: rawResolvedValue
  }
}) {
  // TS isn't able to infer the type of the field name for inputFieldValue()
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  const resolvedValue = rawResolvedValue;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  const fieldName = rawFieldName;
  describe('non-customized field w/ an upgrade (AAB)', () => {
    it('upgrades rule to merged value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: initial,
          target: upgrade,
          merged: upgrade
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.StockValueCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NONE
      });
      const {
        getByRole
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeToMergedValue();
    });
    it('upgrades rule to resolved value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: initial,
          target: upgrade,
          merged: upgrade
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.StockValueCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NONE
      });
      const {
        getByRole,
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      (0, _rule_upgrade_helpers.toggleFieldAccordion)(fieldUpgradeWrapper);
      (0, _rule_upgrade_helpers.switchToFieldEdit)(fieldUpgradeWrapper);
      await (0, _set_field_value.inputFieldValue)(fieldUpgradeWrapper, {
        fieldName,
        value: resolvedValue
      });
      await (0, _rule_upgrade_helpers.saveFieldValue)(fieldUpgradeWrapper);
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeWithResolvedFieldValue(fieldName, resolvedValue);
    });
  });
  describe('customized field w/o an upgrade (ABA)', () => {
    it('upgrades rule to merged value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueNoUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NONE
      });
      const {
        getByRole
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeToMergedValue();
    });
    it('upgrades rule to resolved value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueNoUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NONE
      });
      const {
        getByRole,
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      (0, _rule_upgrade_helpers.toggleFieldAccordion)(fieldUpgradeWrapper);
      (0, _rule_upgrade_helpers.switchToFieldEdit)(fieldUpgradeWrapper);
      await (0, _set_field_value.inputFieldValue)(fieldUpgradeWrapper, {
        fieldName,
        value: resolvedValue
      });
      await (0, _rule_upgrade_helpers.saveFieldValue)(fieldUpgradeWrapper);
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeWithResolvedFieldValue(fieldName, resolvedValue);
    });
  });
  describe('customized field w/ the matching upgrade (ABB)', () => {
    it('upgrades rule to merged value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: upgrade,
          target: upgrade,
          merged: upgrade
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueSameUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NONE
      });
      const {
        getByRole
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeToMergedValue();
    });
    it('upgrades rule to resolved value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: upgrade,
          target: upgrade,
          merged: upgrade
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueSameUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NONE
      });
      const {
        getByRole,
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      (0, _rule_upgrade_helpers.toggleFieldAccordion)(fieldUpgradeWrapper);
      (0, _rule_upgrade_helpers.switchToFieldEdit)(fieldUpgradeWrapper);
      await (0, _set_field_value.inputFieldValue)(fieldUpgradeWrapper, {
        fieldName,
        value: resolvedValue
      });
      await (0, _rule_upgrade_helpers.saveFieldValue)(fieldUpgradeWrapper);
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeWithResolvedFieldValue(fieldName, resolvedValue);
    });
  });
  describe('customized field w/ an upgrade resulting in a solvable conflict (ABC)', () => {
    it('upgrades rule to suggested value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.SOLVABLE
      });
      const {
        getByRole,
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      await (0, _rule_upgrade_helpers.acceptSuggestedFieldValue)(fieldUpgradeWrapper);
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeWithResolvedFieldValue(fieldName, customized);
    });
    it('upgrades rule to resolved value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.SOLVABLE
      });
      const {
        getByRole,
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      (0, _rule_upgrade_helpers.switchToFieldEdit)(fieldUpgradeWrapper);
      await (0, _set_field_value.inputFieldValue)(fieldUpgradeWrapper, {
        fieldName,
        value: resolvedValue
      });
      await (0, _rule_upgrade_helpers.saveAndAcceptFieldValue)(fieldUpgradeWrapper);
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeWithResolvedFieldValue(fieldName, resolvedValue);
    });
  });
  describe('customized field w/ an upgrade resulting in a non-solvable conflict (ABC)', () => {
    it('upgrades rule to suggested value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NON_SOLVABLE
      });
      const {
        getByRole,
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      await (0, _rule_upgrade_helpers.saveAndAcceptFieldValue)(fieldUpgradeWrapper);
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeWithResolvedFieldValue(fieldName, customized);
    });
    it('upgrades rule to resolved value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NON_SOLVABLE
      });
      const {
        getByRole,
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      await (0, _set_field_value.inputFieldValue)(fieldUpgradeWrapper, {
        fieldName,
        value: resolvedValue
      });
      await (0, _rule_upgrade_helpers.saveAndAcceptFieldValue)(fieldUpgradeWrapper);
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeWithResolvedFieldValue(fieldName, resolvedValue);
    });
  });
  describe('missing base version - customized field w/ an upgrade resulted in a solvable conflict (-AB)', () => {
    it('upgrades rule to suggested value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.MissingBaseCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.SOLVABLE
      });
      const {
        getByRole,
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      await (0, _rule_upgrade_helpers.acceptSuggestedFieldValue)(fieldUpgradeWrapper);
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeWithResolvedFieldValue(fieldName, customized);
    });
    it('upgrades rule to resolved value', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.MissingBaseCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.SOLVABLE
      });
      const {
        getByRole,
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      (0, _rule_upgrade_helpers.switchToFieldEdit)(fieldUpgradeWrapper);
      await (0, _set_field_value.inputFieldValue)(fieldUpgradeWrapper, {
        fieldName,
        value: resolvedValue
      });
      await (0, _rule_upgrade_helpers.saveAndAcceptFieldValue)(fieldUpgradeWrapper);
      await clickUpgradeRuleButton(getByRole('dialog'));
      expectRuleUpgradeWithResolvedFieldValue(fieldName, resolvedValue);
    });
  });
}
async function clickUpgradeRuleButton(wrapper) {
  const upgradeRuleButton = (0, _react.within)(wrapper).getByRole('button', {
    name: 'Update rule'
  });
  expect(upgradeRuleButton).toBeVisible();
  await (0, _react.waitFor)(() => expect(upgradeRuleButton).toBeEnabled(), {
    timeout: 500
  });
  await (0, _react.act)(async () => {
    _react.fireEvent.click(upgradeRuleButton);
  });
}
function expectRuleUpgradeToMergedValue() {
  const body = (0, _rule_upgrade_flyout.extractSingleKibanaFetchBodyBy)({
    path: _detection_engine.PERFORM_RULE_UPGRADE_URL,
    method: 'POST'
  });
  expect(body).toMatchObject({
    mode: 'SPECIFIC_RULES',
    rules: [{
      rule_id: 'test-rule',
      revision: 1,
      fields: {}
    }],
    pick_version: 'MERGED'
  });
}
function expectRuleUpgradeWithResolvedFieldValue(fieldName, value) {
  const body = (0, _rule_upgrade_flyout.extractSingleKibanaFetchBodyBy)({
    path: _detection_engine.PERFORM_RULE_UPGRADE_URL,
    method: 'POST'
  });
  expect(body).toMatchObject({
    mode: 'SPECIFIC_RULES',
    rules: [{
      rule_id: 'test-rule',
      revision: 1,
      fields: {
        [fieldName]: (0, _lodash.omitBy)({
          pick_version: 'RESOLVED',
          resolved_value: value
        }, _lodash.isUndefined)
      }
    }],
    pick_version: 'MERGED'
  });
}