"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEnsureAuthorizedActionResult = getEnsureAuthorizedActionResult;
exports.isAuthorizedInAllSpaces = isAuthorizedInAllSpaces;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Helper function that, given an `CheckAuthorizationResult`, checks to see what spaces the user is authorized to perform a given action for
 * the given object type.
 *
 * Only exported for unit testing purposes.
 *
 * @param {string} objectType the object type to check.
 * @param {T} action the action to check.
 * @param {AuthorizationTypeMap<A>} typeMap the typeMap from an CheckAuthorizationResult.
 */
function getEnsureAuthorizedActionResult(objectType, action, typeMap) {
  var _typeMap$get, _record$action;
  const record = (_typeMap$get = typeMap.get(objectType)) !== null && _typeMap$get !== void 0 ? _typeMap$get : {};
  return (_record$action = record[action]) !== null && _record$action !== void 0 ? _record$action : {
    authorizedSpaces: []
  };
}

/**
 * Helper function that, given an `CheckAuthorizationResult`, ensures that the user is authorized to perform a given action for the given
 * object type in the given spaces.
 *
 * @param objectType The object type to check.
 * @param action The action to check.
 * @param spaces The spaces to check.
 * @param typeMap The typeMap from a CheckAuthorizationResult.
 */
function isAuthorizedInAllSpaces(objectType, action, spaces, typeMap) {
  const actionResult = getEnsureAuthorizedActionResult(objectType, action, typeMap);
  const {
    authorizedSpaces,
    isGloballyAuthorized
  } = actionResult;
  const authorizedSpacesSet = new Set(authorizedSpaces);
  return isGloballyAuthorized || spaces.every(space => authorizedSpacesSet.has(space));
}