"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchApi = fetchApi;
exports.parseDataStream = parseDataStream;
var _uuid = require("uuid");
var _stream = require("./stream");
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchApi({
  api,
  messages,
  body,
  headers,
  abortController,
  handleFailure,
  onUpdate
}) {
  var _abortController;
  const requestInit = {
    method: 'POST',
    body: JSON.stringify({
      messages,
      ...body
    }),
    headers: {
      'Content-Type': 'application/json',
      ...headers
    },
    signal: abortController === null || abortController === void 0 ? void 0 : (_abortController = abortController()) === null || _abortController === void 0 ? void 0 : _abortController.signal
  };
  const apiRequest = typeof api === 'string' ? fetch(api, requestInit) : api(requestInit);
  const apiResponse = await apiRequest.catch(async error => {
    let errorMessage = 'Failed to fetch the chat messages';
    if (error.response) {
      var _await$error$response, _await$error$response2, _error$response;
      errorMessage = (_await$error$response = (_await$error$response2 = await ((_error$response = error.response) === null || _error$response === void 0 ? void 0 : _error$response.json())) === null || _await$error$response2 === void 0 ? void 0 : _await$error$response2.message) !== null && _await$error$response !== void 0 ? _await$error$response : 'Failed to fetch the chat response.';
    }
    handleFailure(errorMessage);
    throw new Error(errorMessage);
  });
  if (!apiResponse.body) {
    throw new Error('The response body is empty.');
  }
  const reader = apiResponse.body.getReader();
  return await parseDataStream({
    reader,
    abortControllerRef: abortController != null ? {
      current: abortController()
    } : undefined,
    handleFailure,
    update: onUpdate
  });
}
function assignAnnotationsToMessage(message, annotations) {
  if (!message || !annotations || !annotations.length) return message;
  return {
    ...message,
    annotations: [...annotations]
  };
}
async function parseDataStream({
  reader,
  abortControllerRef,
  update,
  handleFailure,
  generateId = _uuid.v4,
  getCurrentDate = () => new Date()
}) {
  const createdAt = getCurrentDate();
  const prefixMap = {};
  let messageAnnotations;
  for await (const {
    type,
    value
  } of (0, _stream.readDataStream)(reader, {
    isAborted: () => (abortControllerRef === null || abortControllerRef === void 0 ? void 0 : abortControllerRef.current) === null
  })) {
    var _messageAnnotations;
    if (type === 'text') {
      if (prefixMap.text) {
        prefixMap.text = {
          ...prefixMap.text,
          content: (prefixMap.text.content || '') + value
        };
      } else {
        prefixMap.text = {
          id: generateId(),
          role: _types.MessageRole.assistant,
          content: value,
          createdAt
        };
      }
    } else if (type === 'error') {
      handleFailure(value);
      break;
    }
    let responseMessage = prefixMap.text;
    if (type === 'message_annotations') {
      if (!messageAnnotations) {
        messageAnnotations = [...value];
      } else {
        messageAnnotations.push(...value);
      }
      responseMessage = assignAnnotationsToMessage(prefixMap.text, messageAnnotations);
    }
    if ((_messageAnnotations = messageAnnotations) !== null && _messageAnnotations !== void 0 && _messageAnnotations.length) {
      const messagePrefixKeys = ['text'];
      messagePrefixKeys.forEach(key => {
        if (prefixMap[key]) {
          prefixMap[key].annotations = [...messageAnnotations];
        }
      });
    }
    const mergedMessages = [responseMessage].filter(Boolean).map(message => ({
      ...assignAnnotationsToMessage(message, messageAnnotations)
    }));
    update(mergedMessages);
  }
  return {
    messages: [prefixMap.text].filter(Boolean)
  };
}