"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isFileHashTransform = exports.createFileHashTransform = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _stream = require("stream");
var _crypto = require("crypto");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class FileHashTransform extends _stream.Transform {
  constructor(algorithm = 'sha256') {
    super();
    (0, _defineProperty2.default)(this, "hash", void 0);
    (0, _defineProperty2.default)(this, "isFinished", false);
    (0, _defineProperty2.default)(this, "hashValue", undefined);
    this.algorithm = algorithm;
    this.hash = (0, _crypto.createHash)(this.algorithm);
    this.once('finish', () => {
      this.isFinished = true;
    });
  }
  _transform(chunk, _, next) {
    if (!Buffer.isBuffer(chunk)) {
      throw new Error(`Received a non-buffer chunk. All chunk must be buffers.`);
    }
    if (chunk !== null) {
      this.hash.update(chunk);
    }
    next(null, chunk);
  }
  getFileHash() {
    if (!this.isFinished) {
      throw new Error('File hash generation not yet complete');
    }
    if (!this.hashValue) {
      this.hashValue = this.hash.digest('hex');
    }
    return {
      algorithm: this.algorithm,
      value: this.hashValue
    };
  }
}

/**
 * Creates a `Transform` that will calculate a Hash based on the data provided by a Readable
 * @param algorithm
 */
const createFileHashTransform = (algorithm = 'sha256') => {
  return new FileHashTransform(algorithm);
};

/**
 * Type guard to check of a given Transform is a `FileHashTransform`
 * @param transform
 */
exports.createFileHashTransform = createFileHashTransform;
const isFileHashTransform = transform => {
  return transform instanceof FileHashTransform;
};
exports.isFileHashTransform = isFileHashTransform;