"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WhereCommandVisitorContext = exports.VisitorContext = exports.TimeseriesCommandVisitorContext = exports.TimeIntervalLiteralExpressionVisitorContext = exports.StatsCommandVisitorContext = exports.SourceExpressionVisitorContext = exports.SortCommandVisitorContext = exports.ShowCommandVisitorContext = exports.SampleCommandVisitorContext = exports.RrfCommandVisitorContext = exports.RowCommandVisitorContext = exports.RerankCommandVisitorContext = exports.RenameCommandVisitorContext = exports.QueryVisitorContext = exports.OrderExpressionVisitorContext = exports.MvExpandCommandVisitorContext = exports.MetaCommandVisitorContext = exports.MapExpressionVisitorContext = exports.MapEntryExpressionVisitorContext = exports.LookupCommandVisitorContext = exports.LiteralExpressionVisitorContext = exports.ListLiteralExpressionVisitorContext = exports.LimitCommandVisitorContext = exports.KeepCommandVisitorContext = exports.JoinCommandVisitorContext = exports.InlineStatsCommandVisitorContext = exports.InlineCastExpressionVisitorContext = exports.IdentifierExpressionVisitorContext = exports.GrokCommandVisitorContext = exports.FunctionCallExpressionVisitorContext = exports.FromCommandVisitorContext = exports.ForkCommandVisitorContext = exports.ExpressionVisitorContext = exports.ExplainCommandVisitorContext = exports.EvalCommandVisitorContext = exports.EnrichCommandVisitorContext = exports.DropCommandVisitorContext = exports.DissectCommandVisitorContext = exports.CompletionCommandVisitorContext = exports.CommandVisitorContext = exports.CommandOptionVisitorContext = exports.ColumnExpressionVisitorContext = exports.ChangePointCommandVisitorContext = void 0;
var _utils = require("./utils");
var _builder = require("../builder");
var _is = require("../ast/is");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/* eslint-disable max-classes-per-file */
// Splitting classes across files runs into issues with circular dependencies
// and makes it harder to understand the code structure.

class VisitorContext {
  constructor(
  /**
   * Global visitor context.
   */
  ctx,
  /**
   * ES|QL AST node which is currently being visited.
   */
  node,
  /**
   * Context of the parent node, from which the current node was reached
   * during the AST traversal.
   */
  parent = null) {
    this.ctx = ctx;
    this.node = node;
    this.parent = parent;
  }
  *visitArguments(input) {
    this.ctx.assertMethodExists('visitExpression');
    for (const arg of this.arguments()) {
      if (!arg) continue;
      if (arg.type === 'option') {
        continue;
      }
      yield this.visitExpression(arg, typeof input === 'function' ? input() : input);
    }
  }
  arguments() {
    const node = this.node;
    if (!(0, _is.isProperNode)(node)) {
      return [];
    }
    const args = [];
    for (const arg of (0, _utils.children)(node)) {
      args.push(arg);
    }
    return args;
  }
  visitArgument(index, input) {
    this.ctx.assertMethodExists('visitExpression');
    const node = this.node;
    if (!(0, _is.isProperNode)(node)) {
      throw new Error('Node does not have arguments');
    }
    let i = 0;
    for (const arg of this.arguments()) {
      if (i === index) {
        return this.visitExpression(arg, input);
      }
      i++;
    }
    throw new Error(`Argument at index ${index} not found`);
  }
  visitExpression(expressionNode, input) {
    return this.ctx.visitExpression(this, expressionNode, input);
  }
  visitCommand(commandNode, input) {
    return this.ctx.visitCommand(this, commandNode, input);
  }
}
exports.VisitorContext = VisitorContext;
class QueryVisitorContext extends VisitorContext {
  *commands() {
    yield* this.node.commands;
  }
  *visitCommands(input) {
    this.ctx.assertMethodExists('visitCommand');
    for (const cmd of this.node.commands) {
      yield this.visitCommand(cmd, input);
    }
  }
}

// Commands --------------------------------------------------------------------
exports.QueryVisitorContext = QueryVisitorContext;
class CommandVisitorContext extends VisitorContext {
  name() {
    return this.node.name.toUpperCase();
  }
  *options() {
    for (const arg of this.node.args) {
      if (!arg || Array.isArray(arg)) {
        continue;
      }
      if (arg.type === 'option') {
        yield arg;
      }
    }
  }
  *visitOptions(input) {
    this.ctx.assertMethodExists('visitCommandOption');
    for (const option of this.options()) {
      const sourceContext = new CommandOptionVisitorContext(this.ctx, option, this);
      const result = this.ctx.methods.visitCommandOption(sourceContext, input);
      yield result;
    }
  }
  *args(option = '') {
    option = option.toLowerCase();
    if (!option) {
      for (const arg of this.node.args) {
        if (!arg) {
          continue;
        }
        if (Array.isArray(arg)) {
          yield arg;
          continue;
        }
        if (arg.type !== 'option') {
          yield arg;
        }
      }
    }
    const optionNode = this.node.args.find(arg => !Array.isArray(arg) && arg && arg.type === 'option' && arg.name === option);
    if (optionNode) {
      yield* optionNode.args;
    }
  }
  *visitArgs(input, option = '') {
    this.ctx.assertMethodExists('visitExpression');
    for (const arg of (0, _utils.singleItems)(this.args(option))) {
      yield this.visitExpression(arg, typeof input === 'function' ? input() : input);
    }
  }
  *visitSources(input) {
    this.ctx.assertMethodExists('visitSourceExpression');
    for (const arg of (0, _utils.singleItems)(this.node.args)) {
      if (arg.type === 'source') {
        const sourceContext = new SourceExpressionVisitorContext(this.ctx, arg, this);
        const result = this.ctx.methods.visitSourceExpression(sourceContext, input);
        yield result;
      }
    }
  }
  visitSubQuery(queryNode) {
    this.ctx.assertMethodExists('visitQuery');
    return this.ctx.visitQuery(this, queryNode, undefined);
  }
  *visitSubQueries() {
    this.ctx.assertMethodExists('visitQuery');
    for (const arg of this.node.args) {
      if (!arg || Array.isArray(arg)) {
        continue;
      }
      if (arg.type === 'query') {
        const result = this.visitSubQuery(arg);
        yield result;
      }
    }
  }
}
exports.CommandVisitorContext = CommandVisitorContext;
class CommandOptionVisitorContext extends VisitorContext {}

// FROM <sources> [ METADATA <columns> ]
exports.CommandOptionVisitorContext = CommandOptionVisitorContext;
class FromCommandVisitorContext extends CommandVisitorContext {
  /**
   * Visit the METADATA part of the FROM command.
   *
   *   FROM <sources> [ METADATA <columns> ]
   *
   * @param input Input object to pass to all "visitColumn" children methods.
   * @returns An iterable of results of all the "visitColumn" visitor methods.
   */
  *visitMetadataColumns(input) {
    this.ctx.assertMethodExists('visitColumnExpression');
    let metadataOption;
    for (const arg of (0, _utils.singleItems)(this.node.args)) {
      if (arg.type === 'option' && arg.name === 'metadata') {
        metadataOption = arg;
        break;
      }
    }
    if (!metadataOption) {
      return;
    }
    for (const arg of (0, _utils.singleItems)(metadataOption.args)) {
      if (arg.type === 'column') {
        const columnContext = new ColumnExpressionVisitorContext(this.ctx, arg, this);
        const result = this.ctx.methods.visitColumnExpression(columnContext, input);
        yield result;
      }
    }
  }
}

// LIMIT <literal>
exports.FromCommandVisitorContext = FromCommandVisitorContext;
class LimitCommandVisitorContext extends CommandVisitorContext {
  /**
   * @returns The first numeric literal argument of the command.
   */
  numericLiteral() {
    const arg = (0, _utils.firstItem)(this.node.args);
    if (arg && arg.type === 'literal' && (arg.literalType === 'integer' || arg.literalType === 'double')) {
      return arg;
    }
  }

  /**
   * @returns The value of the first numeric literal argument of the command.
   */
  numeric() {
    const literal = this.numericLiteral();
    return literal === null || literal === void 0 ? void 0 : literal.value;
  }
  setLimit(value) {
    const literalNode = _builder.Builder.expression.literal.numeric({
      value,
      literalType: 'integer'
    });
    this.node.args = [literalNode];
  }
}

// EXPLAIN <query>
exports.LimitCommandVisitorContext = LimitCommandVisitorContext;
class ExplainCommandVisitorContext extends CommandVisitorContext {}

// ROW <columns>
exports.ExplainCommandVisitorContext = ExplainCommandVisitorContext;
class RowCommandVisitorContext extends CommandVisitorContext {}

// TS
exports.RowCommandVisitorContext = RowCommandVisitorContext;
class TimeseriesCommandVisitorContext extends CommandVisitorContext {}

// SHOW <identifier>
exports.TimeseriesCommandVisitorContext = TimeseriesCommandVisitorContext;
class ShowCommandVisitorContext extends CommandVisitorContext {}

// META <identifier>
exports.ShowCommandVisitorContext = ShowCommandVisitorContext;
class MetaCommandVisitorContext extends CommandVisitorContext {}

// EVAL <columns>
exports.MetaCommandVisitorContext = MetaCommandVisitorContext;
class EvalCommandVisitorContext extends CommandVisitorContext {}

// STATS <columns> [ BY <columns> ]
exports.EvalCommandVisitorContext = EvalCommandVisitorContext;
class StatsCommandVisitorContext extends CommandVisitorContext {}

// INLINESTATS <columns> [ BY <columns> ]
exports.StatsCommandVisitorContext = StatsCommandVisitorContext;
class InlineStatsCommandVisitorContext extends CommandVisitorContext {}

// LOOKUP <source> ON <column>
exports.InlineStatsCommandVisitorContext = InlineStatsCommandVisitorContext;
class LookupCommandVisitorContext extends CommandVisitorContext {}

// KEEP <columns>
exports.LookupCommandVisitorContext = LookupCommandVisitorContext;
class KeepCommandVisitorContext extends CommandVisitorContext {}

// SORT <columns>
exports.KeepCommandVisitorContext = KeepCommandVisitorContext;
class SortCommandVisitorContext extends CommandVisitorContext {}

// WHERE <expression>
exports.SortCommandVisitorContext = SortCommandVisitorContext;
class WhereCommandVisitorContext extends CommandVisitorContext {}

// DROP <columns>
exports.WhereCommandVisitorContext = WhereCommandVisitorContext;
class DropCommandVisitorContext extends CommandVisitorContext {}

// RENAME <column> AS <column>
exports.DropCommandVisitorContext = DropCommandVisitorContext;
class RenameCommandVisitorContext extends CommandVisitorContext {}

// DISSECT <column> <string> [ APPEND_SEPARATOR = <string> ]
exports.RenameCommandVisitorContext = RenameCommandVisitorContext;
class DissectCommandVisitorContext extends CommandVisitorContext {}

// GROK <column> <string>
exports.DissectCommandVisitorContext = DissectCommandVisitorContext;
class GrokCommandVisitorContext extends CommandVisitorContext {}

// ENRICH <column> [ ON <column> ] [ WITH <columns> ]
exports.GrokCommandVisitorContext = GrokCommandVisitorContext;
class EnrichCommandVisitorContext extends CommandVisitorContext {}

// MV_EXPAND <column>
exports.EnrichCommandVisitorContext = EnrichCommandVisitorContext;
class MvExpandCommandVisitorContext extends CommandVisitorContext {}

// <LOOKUP | LEFT | RIGHT> JOIN <target> ON <condition>
exports.MvExpandCommandVisitorContext = MvExpandCommandVisitorContext;
class JoinCommandVisitorContext extends CommandVisitorContext {}

// RERANK <query> ON field [, field ...] WITH <inference-id>
exports.JoinCommandVisitorContext = JoinCommandVisitorContext;
class RerankCommandVisitorContext extends CommandVisitorContext {}

// CHANGE_POINT <value> [ ON <key> ] [ AS <targetType>, <targetPvalue> ]
exports.RerankCommandVisitorContext = RerankCommandVisitorContext;
class ChangePointCommandVisitorContext extends CommandVisitorContext {}

// FORK (COMMAND ... [| COMMAND ...]) [(COMMAND ... [| COMMAND ...])]
exports.ChangePointCommandVisitorContext = ChangePointCommandVisitorContext;
class ForkCommandVisitorContext extends CommandVisitorContext {}

// COMPLETION
exports.ForkCommandVisitorContext = ForkCommandVisitorContext;
class CompletionCommandVisitorContext extends CommandVisitorContext {}

// SAMPLE <probability> [SEED <seed>]
exports.CompletionCommandVisitorContext = CompletionCommandVisitorContext;
class SampleCommandVisitorContext extends CommandVisitorContext {}

// RRF
exports.SampleCommandVisitorContext = SampleCommandVisitorContext;
class RrfCommandVisitorContext extends CommandVisitorContext {}

// Expressions -----------------------------------------------------------------
exports.RrfCommandVisitorContext = RrfCommandVisitorContext;
class ExpressionVisitorContext extends VisitorContext {}
exports.ExpressionVisitorContext = ExpressionVisitorContext;
class ColumnExpressionVisitorContext extends VisitorContext {}
exports.ColumnExpressionVisitorContext = ColumnExpressionVisitorContext;
class SourceExpressionVisitorContext extends VisitorContext {}
exports.SourceExpressionVisitorContext = SourceExpressionVisitorContext;
class FunctionCallExpressionVisitorContext extends VisitorContext {
  /**
   * @returns Returns a printable uppercase function name or operator.
   */
  operator() {
    const operator = this.node.name;
    switch (operator) {
      case 'note_like':
        {
          return 'NOT LIKE';
        }
      case 'not_rlike':
        {
          return 'NOT RLIKE';
        }
    }
    return operator.toUpperCase();
  }
}
exports.FunctionCallExpressionVisitorContext = FunctionCallExpressionVisitorContext;
class LiteralExpressionVisitorContext extends ExpressionVisitorContext {}
exports.LiteralExpressionVisitorContext = LiteralExpressionVisitorContext;
class ListLiteralExpressionVisitorContext extends ExpressionVisitorContext {
  *visitElements(input) {
    this.ctx.assertMethodExists('visitExpression');
    for (const value of this.node.values) {
      yield this.visitExpression(value, typeof input === 'function' ? input() : input);
    }
  }
}
exports.ListLiteralExpressionVisitorContext = ListLiteralExpressionVisitorContext;
class TimeIntervalLiteralExpressionVisitorContext extends ExpressionVisitorContext {}
exports.TimeIntervalLiteralExpressionVisitorContext = TimeIntervalLiteralExpressionVisitorContext;
class InlineCastExpressionVisitorContext extends ExpressionVisitorContext {
  value() {
    this.ctx.assertMethodExists('visitExpression');
    const value = (0, _utils.firstItem)([this.node.value]);
    return value;
  }
  visitValue(input) {
    this.ctx.assertMethodExists('visitExpression');
    return this.visitExpression(this.value(), input);
  }
}
exports.InlineCastExpressionVisitorContext = InlineCastExpressionVisitorContext;
class OrderExpressionVisitorContext extends VisitorContext {}
exports.OrderExpressionVisitorContext = OrderExpressionVisitorContext;
class IdentifierExpressionVisitorContext extends VisitorContext {}
exports.IdentifierExpressionVisitorContext = IdentifierExpressionVisitorContext;
class MapExpressionVisitorContext extends VisitorContext {
  *visitEntries(input) {
    this.ctx.assertMethodExists(['visitExpression', 'visitMapEntryExpression']);
    for (const value of this.node.entries) {
      yield this.visitExpression(value, typeof input === 'function' ? input() : input);
    }
  }
  *visitArguments(input) {
    return yield* this.visitEntries(input);
  }
}
exports.MapExpressionVisitorContext = MapExpressionVisitorContext;
class MapEntryExpressionVisitorContext extends VisitorContext {
  key() {
    return this.node.key;
  }
  value() {
    return this.node.value;
  }
  visitKey(input) {
    this.ctx.assertMethodExists('visitExpression');
    return this.visitExpression(this.key(), input);
  }
  visitValue(input) {
    this.ctx.assertMethodExists('visitExpression');
    return this.visitExpression(this.value(), input);
  }
}
exports.MapEntryExpressionVisitorContext = MapEntryExpressionVisitorContext;