"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CachedFetchModelsApiLogic = void 0;
var _kea = require("kea");
var _lodash = require("lodash");
var _api = require("../../../../../common/types/api");
var _fetch_models_api_logic = require("./fetch_models_api_logic");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FETCH_MODELS_POLLING_DURATION = 5000; // 5 seconds
const FETCH_MODELS_POLLING_DURATION_ON_FAILURE = 30000; // 30 seconds

const CachedFetchModelsApiLogic = exports.CachedFetchModelsApiLogic = (0, _kea.kea)({
  actions: {
    clearPollTimeout: true,
    createPollTimeout: duration => ({
      duration
    }),
    setTimeoutId: id => ({
      id
    }),
    startPolling: true,
    stopPolling: true
  },
  connect: {
    actions: [_fetch_models_api_logic.FetchModelsApiLogic, ['apiSuccess', 'apiError', 'apiReset', 'makeRequest']],
    values: [_fetch_models_api_logic.FetchModelsApiLogic, ['data', 'status']]
  },
  events: ({
    values
  }) => ({
    beforeUnmount: () => {
      if (values.pollTimeoutId) {
        clearTimeout(values.pollTimeoutId);
      }
    }
  }),
  listeners: ({
    actions,
    values
  }) => ({
    apiError: () => {
      if (values.pollTimeoutId) {
        actions.createPollTimeout(FETCH_MODELS_POLLING_DURATION_ON_FAILURE);
      }
    },
    apiSuccess: () => {
      if (values.pollTimeoutId) {
        actions.createPollTimeout(FETCH_MODELS_POLLING_DURATION);
      }
    },
    createPollTimeout: ({
      duration
    }) => {
      if (values.pollTimeoutId) {
        clearTimeout(values.pollTimeoutId);
      }
      const timeoutId = setTimeout(() => {
        actions.makeRequest({});
      }, duration);
      actions.setTimeoutId(timeoutId);
    },
    startPolling: () => {
      if (values.pollTimeoutId) {
        clearTimeout(values.pollTimeoutId);
      }
      actions.makeRequest({});
      actions.createPollTimeout(FETCH_MODELS_POLLING_DURATION);
    },
    stopPolling: () => {
      if (values.pollTimeoutId) {
        clearTimeout(values.pollTimeoutId);
      }
      actions.clearPollTimeout();
    }
  }),
  path: ['enterprise_search', 'content', 'api', 'fetch_models_api_wrapper'],
  reducers: {
    modelsData: [null, {
      apiReset: () => null,
      apiSuccess: (currentState, newState) => (0, _lodash.isEqual)(currentState, newState) ? currentState : newState
    }],
    pollTimeoutId: [null, {
      clearPollTimeout: () => null,
      setTimeoutId: (_, {
        id
      }) => id
    }]
  },
  selectors: ({
    selectors
  }) => ({
    isInitialLoading: [() => [selectors.status, selectors.modelsData], (status, modelsData) => status === _api.Status.IDLE || modelsData === null && status === _api.Status.LOADING]
  })
});