"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getApiKeyManager = void 0;
var _fake_kibana_request = require("@kbn/security-plugin/server/authentication/api_keys/fake_kibana_request");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _saved_object = require("./saved_object");
var _saved_objects = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getApiKeyManager = deps => {
  return {
    generate: () => generate(deps),
    getClient: () => getClient(deps),
    getRequestFromApiKey
  };
};
exports.getApiKeyManager = getApiKeyManager;
const generate = async deps => {
  const {
    core,
    encryptedSavedObjects,
    request,
    namespace
  } = deps;
  if (!encryptedSavedObjects) {
    throw new Error('Unable to create API key. Ensure encrypted Saved Object client is enabled in this environment.');
  } else if (!request) {
    throw new Error('Unable to create API key due to invalid request');
  } else {
    const apiKey = await generateAPIKey(request, deps);
    const soClient = core.savedObjects.getScopedClient(request, {
      includedHiddenTypes: [_saved_object.PrivilegeMonitoringApiKeyType.name, _saved_objects.monitoringEntitySourceType.name]
    });
    await soClient.create(_saved_object.PrivilegeMonitoringApiKeyType.name, apiKey, {
      id: (0, _saved_object.getPrivmonEncryptedSavedObjectId)(namespace),
      overwrite: true,
      managed: true
    });
  }
};
const getApiKey = async deps => {
  if (!deps.encryptedSavedObjects) {
    throw Error('Unable to retrieve API key. Ensure encrypted Saved Object client is enabled in this environment.');
  }
  try {
    const encryptedSavedObjectsClient = deps.encryptedSavedObjects.getClient({
      includedHiddenTypes: [_saved_object.PrivilegeMonitoringApiKeyType.name]
    });
    return (await encryptedSavedObjectsClient.getDecryptedAsInternalUser(_saved_object.PrivilegeMonitoringApiKeyType.name, (0, _saved_object.getPrivmonEncryptedSavedObjectId)(deps.namespace), {
      namespace: deps.namespace
    })).attributes;
  } catch (err) {
    if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      return undefined;
    }
    throw err;
  }
};
const getRequestFromApiKey = async apiKey => {
  return (0, _fake_kibana_request.getFakeKibanaRequest)({
    id: apiKey.id,
    api_key: apiKey.apiKey
  });
};
const getClient = async deps => {
  const apiKey = await getApiKey(deps);
  if (!apiKey) return undefined;
  const fakeRequest = (0, _fake_kibana_request.getFakeKibanaRequest)({
    id: apiKey.id,
    api_key: apiKey.apiKey
  });
  const clusterClient = deps.core.elasticsearch.client.asScoped(fakeRequest);
  return {
    clusterClient
  };
};
const generateAPIKey = async (req, deps) => {
  deps.logger.info('Generating Privmon API key');
  const apiKey = await deps.security.authc.apiKeys.grantAsInternalUser(req, {
    name: 'Privilege Monitoring API key',
    /**
     * Intentionally passing empty array - generates a snapshot (empty object).
     * Due to not knowing what index pattern changes customer may make to index list.
     *
     * - If the customer later adds new indices they *do* have access to, the key will still function.
     * - If they add indices they *don't* have access to, they will need to reinitialize once their access is elevated.
     */
    role_descriptors: {},
    metadata: {
      description: 'API key used to manage the resources in the privilege monitoring engine'
    }
  });
  if (apiKey !== null) {
    return {
      id: apiKey.id,
      name: apiKey.name,
      apiKey: apiKey.api_key
    };
  }
};