"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createEventSignal = void 0;
var _constants = require("../../../../../../common/constants");
var _build_threat_mapping_filter = require("./build_threat_mapping_filter");
var _get_filter = require("../../utils/get_filter");
var _search_after_bulk_create = require("../../utils/search_after_bulk_create");
var _reason_formatters = require("../../utils/reason_formatters");
var _get_signals_map_from_threat_index = require("./get_signals_map_from_threat_index");
var _search_after_bulk_create_suppressed_alerts = require("../../utils/search_after_bulk_create_suppressed_alerts");
var _threat_enrichment_factory = require("./threat_enrichment_factory");
var _utils = require("./utils");
var _get_is_alert_suppression_active = require("../../utils/get_is_alert_suppression_active");
var _validate_complete_threat_matches = require("./validate_complete_threat_matches");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createEventSignal = async ({
  sharedParams,
  currentResult,
  currentEventList,
  eventsTelemetry,
  filters,
  services,
  wrapSuppressedHits,
  threatFilters,
  threatPitId,
  reassignThreatPitId,
  allowedFieldsForTermsQuery,
  threatMatchedFields,
  inputIndexFields,
  threatIndexFields,
  sortOrder = 'desc',
  isAlertSuppressionActive
}) => {
  var _sharedParams$complet, _threatFiltersFromEve;
  const {
    ruleExecutionLogger,
    exceptionFilter,
    inputIndex,
    completeRule: {
      ruleParams: {
        threatMapping,
        type,
        language,
        query,
        savedId
      }
    }
  } = sharedParams;
  const threatIndicatorPath = (_sharedParams$complet = sharedParams.completeRule.ruleParams.threatIndicatorPath) !== null && _sharedParams$complet !== void 0 ? _sharedParams$complet : _constants.DEFAULT_INDICATOR_SOURCE_PATH;
  const threatFiltersFromEvents = (0, _build_threat_mapping_filter.buildThreatMappingFilter)({
    threatMapping,
    threatList: currentEventList,
    entryKey: 'field',
    allowedFieldsForTermsQuery
  });
  if (!threatFiltersFromEvents.query || ((_threatFiltersFromEve = threatFiltersFromEvents.query) === null || _threatFiltersFromEve === void 0 ? void 0 : _threatFiltersFromEve.bool.should.length) === 0) {
    // empty event list and we do not want to return everything as being
    // a hit so opt to return the existing result.
    ruleExecutionLogger.debug('Indicator items are empty after filtering for missing data, returning without attempting a match');
    return currentResult;
  } else {
    var _threatFiltersFromEve2;
    const threatSearchParams = {
      sharedParams,
      esClient: services.scopedClusterClient.asCurrentUser,
      threatFilters: [...threatFilters, threatFiltersFromEvents],
      threatListConfig: {
        _source: threatMatchedFields.threat,
        fields: undefined
      },
      pitId: threatPitId,
      reassignPitId: reassignThreatPitId,
      indexFields: threatIndexFields
    };
    let signalsQueryMap;
    try {
      signalsQueryMap = await (0, _get_signals_map_from_threat_index.getSignalsQueryMapFromThreatIndex)({
        threatSearchParams,
        eventsCount: currentEventList.length,
        signalValueMap: (0, _utils.getSignalValueMap)({
          eventList: currentEventList,
          threatMatchedFields
        }),
        termsQueryAllowed: true
      });
    } catch (exc) {
      // we receive an error if the event list count < threat list count
      // which puts us into the create_event_signal which differs from create threat signal
      // in that we call getSignalsQueryMapFromThreatIndex which can *throw* an error
      // rather than *return* one.
      if (exc.message.includes(_utils.MANY_NESTED_CLAUSES_ERR) || exc.message.includes(_utils.FAILED_CREATE_QUERY_MAX_CLAUSE)) {
        currentResult.errors.push(exc.message);
        return currentResult;
      } else {
        throw exc;
      }
    }
    const {
      matchedEvents,
      skippedIds
    } = (0, _validate_complete_threat_matches.validateCompleteThreatMatches)(signalsQueryMap, threatMapping);
    if (skippedIds.length > 0) {
      ruleExecutionLogger.debug(`Skipping not matched documents: ${skippedIds.join(', ')}`);
    }
    const ids = Array.from(matchedEvents.keys());
    const indexFilter = {
      query: {
        bool: {
          filter: {
            ids: {
              values: ids
            }
          }
        }
      }
    };
    const esFilter = await (0, _get_filter.getFilter)({
      type,
      filters: [...filters, indexFilter],
      language,
      query,
      savedId,
      services,
      index: inputIndex,
      exceptionFilter,
      fields: inputIndexFields,
      loadFields: true
    });
    ruleExecutionLogger.debug(`${ids === null || ids === void 0 ? void 0 : ids.length} matched signals found`);
    const enrichment = (0, _threat_enrichment_factory.threatEnrichmentFactory)({
      signalsQueryMap: matchedEvents,
      threatIndicatorPath,
      threatFilters,
      threatSearchParams
    });
    let createResult;
    const searchAfterBulkCreateParams = {
      sharedParams,
      buildReasonMessage: _reason_formatters.buildReasonMessageForThreatMatchAlert,
      enrichment,
      eventsTelemetry,
      filter: esFilter,
      services,
      sortOrder,
      trackTotalHits: false
    };
    if (isAlertSuppressionActive && (0, _get_is_alert_suppression_active.alertSuppressionTypeGuard)(sharedParams.completeRule.ruleParams.alertSuppression)) {
      createResult = await (0, _search_after_bulk_create_suppressed_alerts.searchAfterAndBulkCreateSuppressedAlerts)({
        ...searchAfterBulkCreateParams,
        wrapSuppressedHits,
        alertSuppression: sharedParams.completeRule.ruleParams.alertSuppression
      });
    } else {
      createResult = await (0, _search_after_bulk_create.searchAfterAndBulkCreate)(searchAfterBulkCreateParams);
    }
    ruleExecutionLogger.debug(`${(_threatFiltersFromEve2 = threatFiltersFromEvents.query) === null || _threatFiltersFromEve2 === void 0 ? void 0 : _threatFiltersFromEve2.bool.should.length} items have completed match checks and the total times to search were ${createResult.searchAfterTimes.length !== 0 ? createResult.searchAfterTimes : '(unknown) '}ms`);
    return createResult;
  }
};
exports.createEventSignal = createEventSignal;