"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.revertPrebuiltRuleHandler = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _error_helpers = require("../../../../../utils/error_helpers");
var _utils = require("../../../routes/utils");
var _prebuilt_rule_assets_client = require("../../logic/rule_assets/prebuilt_rule_assets_client");
var _bulk_actions_response = require("../../../rule_management/api/rules/bulk_actions/bulk_actions_response");
var _zip_rule_versions = require("../../logic/rule_versions/zip_rule_versions");
var _revert_prebuilt_rules = require("../../logic/rule_objects/revert_prebuilt_rules");
var _get_concurrrency_errors = require("./get_concurrrency_errors");
var _filter_out_non_revertable_rules = require("./filter_out_non_revertable_rules");
var _get_rule_by_id = require("../../../rule_management/logic/detection_rules_client/methods/get_rule_by_id");
var _utils2 = require("../../../rule_management/utils/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const revertPrebuiltRuleHandler = async (context, request, response) => {
  const siemResponse = (0, _utils.buildSiemResponse)(response);
  const {
    id,
    revision,
    version
  } = request.body;
  try {
    const ctx = await context.resolve(['core', 'alerting', 'securitySolution']);
    const soClient = ctx.core.savedObjects.client;
    const rulesClient = await ctx.alerting.getRulesClient();
    const detectionRulesClient = ctx.securitySolution.getDetectionRulesClient();
    const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(soClient);
    const concurrencySet = {
      [id]: {
        revision,
        version
      }
    };
    const updated = [];
    const errors = [];
    const ruleResponse = await (0, _get_rule_by_id.getRuleById)({
      rulesClient,
      id
    });
    if (!ruleResponse) {
      errors.push((0, _utils2.createBulkActionError)({
        id,
        message: `Cannot find rule with id: ${id}`,
        statusCode: 404
      }));

      // Return early as there's no reason to continue if we can't find the rule
      return buildRuleReversionResponse(response, {
        updated,
        skipped: [],
        errors
      });
    }
    const {
      rulesToRevert,
      skipped
    } = (0, _filter_out_non_revertable_rules.filterOutNonRevertableRules)([ruleResponse]);
    const prebuiltRuleAssets = await ruleAssetsClient.fetchAssetsByVersion(rulesToRevert);
    const ruleVersionsMap = (0, _zip_rule_versions.zipRuleVersions)(rulesToRevert, [], prebuiltRuleAssets); // We use base versions as target param as we are reverting rules
    const revertableRules = [];
    rulesToRevert.forEach(rule => {
      const ruleVersions = ruleVersionsMap.get(rule.rule_id);
      const currentVersion = ruleVersions === null || ruleVersions === void 0 ? void 0 : ruleVersions.current;
      const baseVersion = ruleVersions === null || ruleVersions === void 0 ? void 0 : ruleVersions.target;
      if (!currentVersion) {
        errors.push((0, _utils2.createBulkActionError)({
          id: rule.id,
          message: `Cannot find rule with id: ${rule.id}`,
          statusCode: 404
        }));
        return;
      }
      if (!baseVersion) {
        errors.push((0, _utils2.createBulkActionError)({
          id: rule.id,
          message: `Cannot find base_version for rule id: ${rule.id}`,
          statusCode: 404
        }));
        return;
      }
      const concurrencyError = (0, _get_concurrrency_errors.getConcurrencyErrors)(concurrencySet[rule.id].revision, concurrencySet[rule.id].version, rule);
      if (concurrencyError) {
        errors.push(concurrencyError);
        return;
      }
      revertableRules.push({
        current: currentVersion,
        target: baseVersion // Use base version as target to revert rule
      });
    });
    const {
      results: revertResults,
      errors: revertErrors
    } = await (0, _revert_prebuilt_rules.revertPrebuiltRules)(detectionRulesClient, revertableRules);
    const formattedUpdateErrors = revertErrors.map(({
      error,
      item
    }) => {
      return {
        message: (0, _error_helpers.getErrorMessage)(error),
        status: (0, _error_helpers.getErrorStatusCode)(error),
        rule: item.current
      };
    });
    errors.push(...formattedUpdateErrors);
    updated.push(...revertResults.map(({
      result
    }) => result));
    return buildRuleReversionResponse(response, {
      updated,
      skipped,
      errors
    });
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    return siemResponse.error({
      body: error.message,
      statusCode: error.statusCode
    });
  }
};

// Similar to `buildBulkResponse` in /bulk_actions_response.ts but the `RevertPrebuiltRulesResponseBody` type has a slightly different return body
// If we extend the revert route this can be folded into the existing buildBuleResponse function
exports.revertPrebuiltRuleHandler = revertPrebuiltRuleHandler;
const buildRuleReversionResponse = (response, {
  updated,
  skipped,
  errors
}) => {
  const numSucceeded = updated.length;
  const numSkipped = skipped.length;
  const numFailed = errors.length;
  const summary = {
    failed: numFailed,
    succeeded: numSucceeded,
    skipped: numSkipped,
    total: numSucceeded + numFailed + numSkipped
  };
  const results = {
    updated,
    skipped,
    created: [],
    deleted: []
  };
  if (numFailed > 0) {
    const message = summary.succeeded > 0 ? 'Rule reversion partially failed' : 'Rule reversion failed';
    return response.custom({
      headers: {
        'content-type': 'application/json'
      },
      body: {
        message,
        status_code: 500,
        attributes: {
          errors: (0, _bulk_actions_response.normalizeErrorResponse)(errors),
          results,
          summary
        }
      },
      statusCode: 500
    });
  }
  const responseBody = {
    success: true,
    rules_count: summary.total,
    attributes: {
      results,
      summary
    }
  };
  return response.ok({
    body: responseBody
  });
};