"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexExplorer = void 0;
var _zod = require("@kbn/zod");
var _list_indices = require("./steps/list_indices");
var _get_mappings = require("./steps/get_mappings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const indexExplorer = async ({
  query,
  indexPattern = '*',
  limit = 1,
  esClient,
  model
}) => {
  const allIndices = await (0, _list_indices.listIndices)({
    pattern: indexPattern,
    esClient
  });
  const selectedIndices = await selectIndices({
    indices: allIndices,
    query,
    model,
    limit
  });
  const mappings = await (0, _get_mappings.getIndexMappings)({
    indices: selectedIndices.map(index => index.indexName),
    esClient
  });
  const relevantIndices = selectedIndices.map(({
    indexName,
    reason
  }) => {
    return {
      indexName,
      reason,
      mappings: mappings[indexName].mappings
    };
  });
  return {
    indices: relevantIndices
  };
};
exports.indexExplorer = indexExplorer;
const selectIndices = async ({
  indices,
  query,
  model,
  limit = 1
}) => {
  const {
    chatModel
  } = model;
  const indexSelectorModel = chatModel.withStructuredOutput(_zod.z.object({
    indices: _zod.z.array(_zod.z.object({
      indexName: _zod.z.string().describe('name of the index'),
      reason: _zod.z.string().describe('brief explanation of why this index could be relevant')
    }))
  }));
  const indexSelectorPrompt = [['user', `You are an AI assistant for the Elasticsearch company.
       based on a natural language query from the user, your task is to select up to ${limit} most relevant indices from a list of indices.

       *The query is:* ${query}

       *List of indices:*
       ${indices.map(index => `- ${index.index}`).join('\n')}

       Based on those information, please return most relevant indices with your reasoning.
       Remember, you should select at maximum ${limit} indices.
       `]];
  const {
    indices: selectedIndices
  } = await indexSelectorModel.invoke(indexSelectorPrompt);
  return selectedIndices;
};