"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerLensVisualizationsCreateAPIRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _boom = require("@hapi/boom");
var _content_management = require("../../../../common/content_management");
var _constants = require("../../constants");
var _v = require("../../../content_management/v1");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerLensVisualizationsCreateAPIRoute = (router, {
  contentManagement
}) => {
  const createRoute = router.post({
    path: `${_constants.PUBLIC_API_PATH}/visualizations`,
    access: _constants.PUBLIC_API_ACCESS,
    enableQueryVersion: true,
    summary: 'Create Lens visualization',
    description: 'Create a new Lens visualization.',
    options: {
      tags: ['oas-tag:Lens'],
      availability: {
        stability: 'experimental'
      }
    },
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on Content Client for authorization'
      }
    }
  });
  createRoute.addVersion({
    version: _constants.PUBLIC_API_VERSION,
    validate: {
      request: {
        body: _configSchema.schema.object({
          options: _v.lensCreateOptionsSchema,
          data: _v.lensAttributesSchema
        })
      },
      response: {
        201: {
          body: () => _v.lensSavedObjectSchema,
          description: 'Created'
        },
        400: {
          description: 'Malformed request'
        },
        401: {
          description: 'Unauthorized'
        },
        403: {
          description: 'Forbidden'
        },
        500: {
          description: 'Internal Server Error'
        }
      }
    }
  }, async (ctx, req, res) => {
    let result;
    const {
      data,
      options
    } = req.body;
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_content_management.CONTENT_ID, _constants.PUBLIC_API_CONTENT_MANAGEMENT_VERSION);
    try {
      ({
        result
      } = await client.create(data, options));
    } catch (error) {
      if ((0, _boom.isBoom)(error) && error.output.statusCode === 403) {
        return res.forbidden();
      }
      return (0, _boom.boomify)(error); // forward unknown error
    }
    return res.created({
      body: result.item
    });
  });
};
exports.registerLensVisualizationsCreateAPIRoute = registerLensVisualizationsCreateAPIRoute;