"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.forceUnenrollAgent = forceUnenrollAgent;
exports.unenrollAgent = unenrollAgent;
exports.unenrollAgents = unenrollAgents;
var _uuid = require("uuid");
var _errors = require("../../errors");
var _constants = require("../../constants");
var _get_current_namespace = require("../spaces/get_current_namespace");
var _agent_namespaces = require("../spaces/agent_namespaces");
var _actions = require("./actions");
var _crud = require("./crud");
var _unenroll_action_runner = require("./unenroll_action_runner");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function unenrollAgentIsAllowed(soClient, esClient, agentId, options) {
  const agentPolicy = await (0, _crud.getAgentPolicyForAgent)(soClient, esClient, agentId);
  if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_managed) {
    throw new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot unenroll ${agentId} from a hosted agent policy ${agentPolicy.id}`);
  }
  if (!(options !== null && options !== void 0 && options.skipAgentlessValidation) && agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.supports_agentless) {
    throw new _errors.FleetError(`Cannot unenroll agentless agent ${agentId}`);
  }
  return true;
}
async function unenrollAgent(soClient, esClient, agentId, options) {
  await (0, _crud.getAgentById)(esClient, soClient, agentId); // throw 404 if agent not in namespace
  if (!(options !== null && options !== void 0 && options.force)) {
    await unenrollAgentIsAllowed(soClient, esClient, agentId, {
      skipAgentlessValidation: options === null || options === void 0 ? void 0 : options.skipAgentlessValidation
    });
  }
  if (options !== null && options !== void 0 && options.revoke) {
    return forceUnenrollAgent(esClient, soClient, agentId);
  }
  const now = new Date().toISOString();
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  await (0, _actions.createAgentAction)(esClient, {
    agents: [agentId],
    created_at: now,
    type: 'UNENROLL',
    namespaces: [currentSpaceId]
  });
  await (0, _crud.updateAgent)(esClient, agentId, {
    unenrollment_started_at: now
  });
}
async function unenrollAgents(soClient, esClient, options) {
  var _options$batchSize, _options$showInactive;
  const spaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  if ('agentIds' in options) {
    const givenAgents = await (0, _crud.getAgents)(esClient, soClient, options);
    return await (0, _unenroll_action_runner.unenrollBatch)(soClient, esClient, givenAgents, {
      ...options,
      spaceId
    });
  }
  const batchSize = (_options$batchSize = options.batchSize) !== null && _options$batchSize !== void 0 ? _options$batchSize : _constants.SO_SEARCH_LIMIT;
  const namespaceFilter = await (0, _agent_namespaces.agentsKueryNamespaceFilter)(spaceId);
  const kuery = namespaceFilter ? `${namespaceFilter} AND ${options.kuery}` : options.kuery;
  const res = await (0, _crud.getAgentsByKuery)(esClient, soClient, {
    kuery,
    showAgentless: options.showAgentless,
    showInactive: (_options$showInactive = options.showInactive) !== null && _options$showInactive !== void 0 ? _options$showInactive : false,
    page: 1,
    perPage: batchSize
  });
  if (res.total <= batchSize) {
    return await (0, _unenroll_action_runner.unenrollBatch)(soClient, esClient, res.agents, {
      ...options,
      spaceId
    });
  } else {
    return await new _unenroll_action_runner.UnenrollActionRunner(esClient, soClient, {
      ...options,
      spaceId,
      batchSize,
      total: res.total
    }, {
      pitId: await (0, _crud.openPointInTime)(esClient)
    }).runActionAsyncTask();
  }
}
async function forceUnenrollAgent(esClient, soClient, agentIdOrAgent) {
  const agent = typeof agentIdOrAgent === 'string' ? await (0, _crud.getAgentById)(esClient, soClient, agentIdOrAgent) : agentIdOrAgent;
  await (0, _unenroll_action_runner.invalidateAPIKeysForAgents)([agent]);
  await (0, _crud.updateAgent)(esClient, agent.id, {
    active: false,
    unenrolled_at: new Date().toISOString()
  });
  await (0, _unenroll_action_runner.updateActionsForForceUnenroll)(esClient, soClient, [agent.id], (0, _uuid.v4)(), 1);
}