"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defaultFleetErrorHandler = void 0;
exports.fleetErrorToResponseOptions = fleetErrorToResponseOptions;
var _boom = require("@hapi/boom");
var _errors = require("../../common/errors");
var _services = require("../services");
var _ = require(".");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// unsure if this is correct. would prefer to use something "official"
// this type is based on BadRequest values observed while debugging https://github.com/elastic/kibana/issues/75862
const getHTTPResponseCode = error => {
  // Bad Request
  if (error instanceof _.PackageInvalidDeploymentMode) {
    return 400;
  }
  if (error instanceof _.PackageFailedVerificationError) {
    return 400;
  }
  if (error instanceof _.PackageOutdatedError) {
    return 400;
  }
  if (error instanceof _.PackageInvalidArchiveError) {
    return 400;
  }
  if (error instanceof _.PackageRemovalError) {
    return 400;
  }
  if (error instanceof _.KibanaSOReferenceError) {
    return 400;
  }
  if (error instanceof _.AgentPolicyInvalidError) {
    return 400;
  }
  if (error instanceof _.AgentRequestInvalidError) {
    return 400;
  }
  if (error instanceof _.PackagePolicyRequestError) {
    return 400;
  }
  if (error instanceof _.PackagePolicyContentPackageError) {
    return 400;
  }
  if (error instanceof _.CustomPackagePolicyNotAllowedForAgentlessError) {
    return 400;
  }
  // Unauthorized
  if (error instanceof _.FleetUnauthorizedError) {
    return 403;
  }
  // Not Found
  if (error instanceof _.FleetNotFoundError) {
    return 404;
  }

  // Conflict
  if (error instanceof _.AgentPolicyNameExistsError) {
    return 409;
  }
  if (error instanceof _.EnrollmentKeyNameExistsError) {
    return 409;
  }
  if (error instanceof _.ConcurrentInstallOperationError) {
    return 409;
  }
  if (error instanceof _.PackageSavedObjectConflictError) {
    return 409;
  }
  if (error instanceof _.PackagePolicyNameExistsError) {
    return 409;
  }
  if (error instanceof _.PackageAlreadyInstalledError) {
    return 409;
  }
  if (error instanceof _.AgentlessPolicyExistsRequestError) {
    return 409;
  }
  // Unsupported Media Type
  if (error instanceof _.PackageUnsupportedMediaTypeError) {
    return 415;
  }
  // Too many requests
  if (error instanceof _.FleetTooManyRequestsError) {
    return 429;
  }
  // Internal Server Error
  if (error instanceof _errors.UninstallTokenError) {
    return 500;
  }
  if (error instanceof _.BundledPackageLocationNotFoundError) {
    return 500;
  }
  if (error instanceof _.PackageESError) {
    return 500;
  }
  if (error instanceof _.RegistryResponseError) {
    // 4xx/5xx's from EPR
    return 500;
  }
  // Bad Gateway
  if (error instanceof _.RegistryConnectionError || error instanceof _.RegistryError) {
    // Connection errors (ie. RegistryConnectionError) / fallback  (RegistryError) from EPR
    return 502;
  }
  if (error instanceof _.FleetErrorWithStatusCode && error.statusCode) {
    return error.statusCode;
  }
  return 400; // Bad Request
};
function shouldRespondWithErrorType(error) {
  if (error instanceof _.OutputInvalidError) {
    return true;
  } else if (error instanceof _.AgentlessAgentCreateOverProvisionnedError) {
    return true;
  } else if (error instanceof _errors.AgentlessAgentCreateFleetUnreachableError) {
    return true;
  }
  return false;
}
function getErrorExtraAttributes(error) {
  if (error instanceof _.AgentlessAgentCreateOverProvisionnedError) {
    var _error$meta;
    return {
      limit: (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : _error$meta.limit
    };
  }
}
function fleetErrorToResponseOptions(error) {
  const logger = _services.appContextService.getLogger();
  // our "expected" errors
  if (error instanceof _.FleetError) {
    // only log the message
    const extraAttributes = getErrorExtraAttributes(error);
    logger.error(error.message);
    return {
      statusCode: getHTTPResponseCode(error),
      body: {
        message: error.message,
        ...(extraAttributes ? {
          attributes: extraAttributes
        } : {}),
        ...(shouldRespondWithErrorType(error) ? {
          attributes: {
            type: error.name,
            ...extraAttributes
          }
        } : {}),
        ...(error.attributes && {
          attributes: {
            ...error.attributes,
            ...extraAttributes
          }
        })
      }
    };
  }

  // handle any older Boom-based errors or the few places our app uses them
  if ((0, _boom.isBoom)(error)) {
    // only log the message
    logger.error(error.output.payload.message);
    return {
      statusCode: error.output.statusCode,
      body: {
        message: error.output.payload.message
      }
    };
  }

  // default response is 500
  logger.error(error);
  return {
    statusCode: 500,
    body: {
      message: error.message
    }
  };
}
const defaultFleetErrorHandler = async ({
  error,
  response
}) => {
  const options = fleetErrorToResponseOptions(error);
  return response.customError(options);
};
exports.defaultFleetErrorHandler = defaultFleetErrorHandler;