"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enableEntityDiscoveryRoute = void 0;
var _entitiesSchema = require("@kbn/entities-schema");
var _zod = require("@kbn/zod");
var _errors = require("../../../common/errors");
var _auth = require("../../lib/auth");
var _built_in = require("../../lib/entities/built_in");
var _install_entity_definition = require("../../lib/entities/install_entity_definition");
var _saved_objects = require("../../saved_objects");
var _create_entity_manager_server_route = require("../create_entity_manager_server_route");
var _start_transforms = require("../../lib/entities/start_transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * @openapi
 * /internal/entities/managed/enablement:
 *   put:
 *     description: Enable managed (built-in) entity discovery.
 *     tags:
 *       - management
 *     parameters:
 *       - in: query
 *         name: installOnly
 *         description: If true, the definition transforms will not be started
 *         required: false
 *         schema:
 *           type: boolean
 *           default: false
 *     responses:
 *       200:
 *         description: OK - Verify result in response body
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               required: success
 *               properties:
 *                 success:
 *                  type: boolean
 *                  example: false
 *                 reason:
 *                  type: string
 *                  example: api_key_service_disabled
 *                 message:
 *                  type: string
 *                  example: API key service is not enabled; try configuring `xpack.security.authc.api_key.enabled` in your elasticsearch config
 *       403:
 *         description: The current user does not have the required permissions to enable entity discovery
 */
const enableEntityDiscoveryRoute = exports.enableEntityDiscoveryRoute = (0, _create_entity_manager_server_route.createEntityManagerServerRoute)({
  endpoint: 'PUT /internal/entities/managed/enablement',
  security: {
    authz: {
      enabled: false,
      reason: 'This endpoint leverages the security plugin to evaluate the privileges needed as part of its core flow'
    }
  },
  params: _zod.z.object({
    query: _entitiesSchema.createEntityDefinitionQuerySchema
  }),
  handler: async ({
    context,
    request,
    response,
    params,
    server,
    logger
  }) => {
    try {
      const apiKeysEnabled = await (0, _auth.checkIfAPIKeysAreEnabled)(server);
      if (!apiKeysEnabled) {
        return response.ok({
          body: {
            success: false,
            reason: _errors.ERROR_API_KEY_SERVICE_DISABLED,
            message: 'API key service is not enabled; try configuring `xpack.security.authc.api_key.enabled` in your elasticsearch config'
          }
        });
      }
      const core = await context.core;
      const esClientAsCurrentUser = core.elasticsearch.client.asCurrentUser;
      const canEnable = await (0, _auth.canEnableEntityDiscovery)(esClientAsCurrentUser);
      if (!canEnable) {
        return response.forbidden({
          body: {
            message: 'Current Kibana user does not have the required permissions to enable entity discovery'
          }
        });
      }
      logger.debug(`Enabling managed entity discovery (installOnly=${params.query.installOnly})`);
      const soClient = core.savedObjects.getClient({
        includedHiddenTypes: [_saved_objects.EntityDiscoveryApiKeyType.name]
      });
      const existingApiKey = await (0, _auth.readEntityDiscoveryAPIKey)(server);
      if (existingApiKey !== undefined) {
        const isValid = await (0, _auth.checkIfEntityDiscoveryAPIKeyIsValid)(server, existingApiKey);
        if (!isValid) {
          await (0, _auth.deleteEntityDiscoveryAPIKey)(soClient);
          await server.security.authc.apiKeys.invalidateAsInternalUser({
            ids: [existingApiKey.id]
          });
        }
      }
      const apiKey = await (0, _auth.generateEntityDiscoveryAPIKey)(server, request);
      if (apiKey === undefined) {
        return response.customError({
          statusCode: 500,
          body: new Error('could not generate entity discovery API key')
        });
      }
      await (0, _auth.saveEntityDiscoveryAPIKey)(soClient, apiKey);
      const esClient = core.elasticsearch.client.asCurrentUser;
      const installedDefinitions = await (0, _install_entity_definition.installBuiltInEntityDefinitions)({
        esClient,
        soClient,
        logger,
        definitions: _built_in.builtInDefinitions
      });
      if (!params.query.installOnly) {
        await Promise.all(installedDefinitions.map(installedDefinition => (0, _start_transforms.startTransforms)(esClient, installedDefinition, logger)));
      }
      logger.debug('Managed entity discovery is enabled');
      return response.ok({
        body: {
          success: true
        }
      });
    } catch (err) {
      logger.error(err);
      return response.customError({
        statusCode: 500,
        body: err
      });
    }
  }
});