"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTabsStorageManager = exports.TABS_LOCAL_STORAGE_KEY = exports.RECENTLY_CLOSED_TABS_LIMIT = void 0;
var _lodash = require("lodash");
var _public = require("@kbn/kibana-utils-plugin/public");
var _constants = require("../../../../common/constants");
var _utils = require("./redux/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const TABS_LOCAL_STORAGE_KEY = exports.TABS_LOCAL_STORAGE_KEY = 'discover.tabs';
const RECENTLY_CLOSED_TABS_LIMIT = exports.RECENTLY_CLOSED_TABS_LIMIT = 50;
const defaultTabsStateInLocalStorage = {
  userId: '',
  spaceId: '',
  openTabs: [],
  closedTabs: []
};
const createTabsStorageManager = ({
  urlStateStorage,
  storage,
  enabled
}) => {
  const urlStateContainer = (0, _public.createStateContainer)({});
  const sessionInfo = {
    userId: '',
    spaceId: ''
  };
  const startUrlSync = ({
    onChanged // can be called when selectedTabId changes in URL to trigger app state change if needed
  }) => {
    if (!enabled) {
      return () => {
        // do nothing
      };
    }
    const {
      start,
      stop
    } = (0, _public.syncState)({
      stateStorage: urlStateStorage,
      stateContainer: {
        ...urlStateContainer,
        set: state => {
          if (state) {
            // syncState utils requires to handle incoming "null" value
            urlStateContainer.set(state);
          }
        }
      },
      storageKey: _constants.TABS_STATE_URL_KEY
    });
    const listener = onChanged ? urlStateContainer.state$.subscribe(state => {
      onChanged(state);
    }) : null;
    start();
    return () => {
      listener === null || listener === void 0 ? void 0 : listener.unsubscribe();
      stop();
    };
  };
  const getSelectedTabIdFromURL = () => {
    var _urlStateStorage$get;
    return (_urlStateStorage$get = urlStateStorage.get(_constants.TABS_STATE_URL_KEY)) === null || _urlStateStorage$get === void 0 ? void 0 : _urlStateStorage$get.tabId;
  };
  const pushSelectedTabIdToUrl = async selectedTabId => {
    const nextState = {
      tabId: selectedTabId
    };
    await urlStateStorage.set(_constants.TABS_STATE_URL_KEY, nextState);
  };
  const toTabStateInStorage = (tabState, getAppState) => {
    const getAppStateForTabWithoutRuntimeState = tabId => getAppState(tabId) || tabState.initialAppState;
    return {
      id: tabState.id,
      label: tabState.label,
      appState: getAppStateForTabWithoutRuntimeState(tabState.id),
      globalState: tabState.lastPersistedGlobalState || tabState.initialGlobalState
    };
  };
  const toRecentlyClosedTabStateInStorage = (tabState, getAppState) => {
    const state = toTabStateInStorage(tabState, getAppState);
    return {
      ...state,
      closedAt: tabState.closedAt
    };
  };
  const getDefinedStateOnly = state => {
    if (!state || !Object.keys(state).length) {
      return undefined;
    }
    return state;
  };
  const toTabState = (tabStateInStorage, defaultTabState) => {
    const appState = getDefinedStateOnly(tabStateInStorage.appState);
    const globalState = getDefinedStateOnly(tabStateInStorage.globalState || defaultTabState.lastPersistedGlobalState);
    return {
      ...defaultTabState,
      ...(0, _lodash.pick)(tabStateInStorage, 'id', 'label'),
      initialAppState: appState,
      initialGlobalState: globalState,
      lastPersistedGlobalState: globalState || {}
    };
  };
  const toRecentlyClosedTabState = (tabStateInStorage, defaultTabState) => ({
    ...toTabState(tabStateInStorage, defaultTabState),
    closedAt: tabStateInStorage.closedAt
  });
  const readFromLocalStorage = () => {
    const storedTabsState = storage.get(TABS_LOCAL_STORAGE_KEY);
    return {
      userId: (storedTabsState === null || storedTabsState === void 0 ? void 0 : storedTabsState.userId) || '',
      spaceId: (storedTabsState === null || storedTabsState === void 0 ? void 0 : storedTabsState.spaceId) || '',
      openTabs: (storedTabsState === null || storedTabsState === void 0 ? void 0 : storedTabsState.openTabs) || [],
      closedTabs: (storedTabsState === null || storedTabsState === void 0 ? void 0 : storedTabsState.closedTabs) || []
    };
  };
  const getNRecentlyClosedTabs = (previousRecentlyClosedTabs, newClosedTabs) => {
    var _latestNRecentlyClose;
    const closedAt = Date.now();
    const newRecentlyClosedTabs = newClosedTabs.map(tab => ({
      ...tab,
      closedAt
    }));
    const newSortedRecentlyClosedTabs = (0, _lodash.orderBy)([...newRecentlyClosedTabs, ...previousRecentlyClosedTabs], 'closedAt', 'desc');
    const latestNRecentlyClosedTabs = newSortedRecentlyClosedTabs.slice(0, RECENTLY_CLOSED_TABS_LIMIT);
    const recentClosedAt = (_latestNRecentlyClose = latestNRecentlyClosedTabs[latestNRecentlyClosedTabs.length - 1]) === null || _latestNRecentlyClose === void 0 ? void 0 : _latestNRecentlyClose.closedAt;
    if (recentClosedAt) {
      // keep other recently closed tabs from the same time point when they were closed
      for (let i = RECENTLY_CLOSED_TABS_LIMIT; i < newSortedRecentlyClosedTabs.length; i++) {
        if (newSortedRecentlyClosedTabs[i].closedAt === recentClosedAt) {
          latestNRecentlyClosedTabs.push(newSortedRecentlyClosedTabs[i]);
        } else {
          break;
        }
      }
    }
    return latestNRecentlyClosedTabs;
  };
  const persistLocally = async ({
    allTabs,
    selectedTabId,
    recentlyClosedTabs
  }, getAppState) => {
    if (!enabled) {
      return;
    }
    await pushSelectedTabIdToUrl(selectedTabId);
    const keptTabIds = {};
    const openTabs = allTabs.map(tab => {
      const tabStateInStorage = toTabStateInStorage(tab, getAppState);
      keptTabIds[tab.id] = true;
      return tabStateInStorage;
    });
    const closedTabs = recentlyClosedTabs.map(tab => {
      const tabStateInStorage = toRecentlyClosedTabStateInStorage(tab, getAppState);
      keptTabIds[tab.id] = true;
      return tabStateInStorage;
    });
    const nextTabsInStorage = {
      userId: sessionInfo.userId,
      spaceId: sessionInfo.spaceId,
      openTabs,
      closedTabs // wil be used for "Recently closed tabs" feature
    };
    storage.set(TABS_LOCAL_STORAGE_KEY, nextTabsInStorage);
  };
  const updateTabStateLocally = (tabId, tabStatePartial) => {
    if (!enabled) {
      return;
    }
    let hasModifications = false;
    const storedTabsState = readFromLocalStorage();
    const updatedTabsState = {
      ...storedTabsState,
      openTabs: storedTabsState.openTabs.map(tab => {
        if (tab.id === tabId) {
          hasModifications = true;
          return {
            ...tab,
            appState: tabStatePartial.appState,
            globalState: tabStatePartial.globalState
          };
        }
        return tab;
      })
    };
    if (hasModifications) {
      storage.set(TABS_LOCAL_STORAGE_KEY, updatedTabsState);
    }
  };
  const loadLocally = ({
    userId,
    spaceId,
    defaultTabState
  }) => {
    const selectedTabId = enabled ? getSelectedTabIdFromURL() : undefined;
    let storedTabsState = enabled ? readFromLocalStorage() : defaultTabsStateInLocalStorage;
    if (storedTabsState.userId !== userId || storedTabsState.spaceId !== spaceId) {
      // if the userId or spaceId has changed, don't read from the local storage
      storedTabsState = {
        ...defaultTabsStateInLocalStorage,
        userId,
        spaceId
      };
    }
    sessionInfo.userId = userId;
    sessionInfo.spaceId = spaceId;
    const openTabs = storedTabsState.openTabs.map(tab => toTabState(tab, defaultTabState));
    const closedTabs = storedTabsState.closedTabs.map(tab => toRecentlyClosedTabState(tab, defaultTabState));
    if (enabled) {
      if (selectedTabId) {
        // restore previously opened tabs
        if (openTabs.find(tab => tab.id === selectedTabId)) {
          return {
            allTabs: openTabs,
            selectedTabId,
            recentlyClosedTabs: closedTabs
          };
        }
        const storedClosedTab = storedTabsState.closedTabs.find(tab => tab.id === selectedTabId);
        if (storedClosedTab) {
          // restore previously closed tabs, for example when only the default tab was shown
          return {
            allTabs: storedTabsState.closedTabs.filter(tab => tab.closedAt === storedClosedTab.closedAt).map(tab => toTabState(tab, defaultTabState)),
            selectedTabId,
            recentlyClosedTabs: getNRecentlyClosedTabs(closedTabs, openTabs)
          };
        }
      }
    }
    const defaultTab = {
      ...defaultTabState,
      ...(0, _utils.createTabItem)([])
    };
    return {
      allTabs: [defaultTab],
      selectedTabId: defaultTab.id,
      recentlyClosedTabs: getNRecentlyClosedTabs(closedTabs, openTabs)
    };
  };
  return {
    startUrlSync,
    persistLocally,
    updateTabStateLocally,
    loadLocally,
    getNRecentlyClosedTabs
  };
};
exports.createTabsStorageManager = createTabsStorageManager;