"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerUpdateScriptedFieldRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _error = require("../../../error");
var _handle_errors = require("../util/handle_errors");
var _schemas = require("../../../schemas");
var _constants = require("../../../constants");
var _schema = require("../../schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registerUpdateScriptedFieldRoute = (router, getStartServices) => {
  router.versioned.post({
    path: '/api/index_patterns/index_pattern/{id}/scripted_field/{name}',
    access: 'public',
    security: {
      authz: {
        requiredPrivileges: ['indexPatterns:manage']
      }
    }
  }).addVersion({
    version: _constants.INITIAL_REST_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            minLength: 1,
            maxLength: 1_000
          }),
          name: _configSchema.schema.string({
            minLength: 1,
            maxLength: 1_000
          })
        }, {
          unknowns: 'allow'
        }),
        body: _configSchema.schema.object({
          field: _configSchema.schema.object({
            ..._schemas.fieldSpecSchemaFields,
            // We need to overwrite the below fields on top of `fieldSpecSchemaFields`,
            // because `name` field must not appear here and other below fields
            // should be possible to not provide `schema.maybe()` instead of
            // them being required with a default value in `fieldSpecSchemaFields`.
            name: _configSchema.schema.never(),
            type: _configSchema.schema.maybe(_configSchema.schema.string({
              maxLength: 1_000
            })),
            searchable: _configSchema.schema.maybe(_configSchema.schema.boolean()),
            aggregatable: _configSchema.schema.maybe(_configSchema.schema.boolean())
          })
        })
      },
      response: {
        200: {
          body: _schema.indexPatternsRuntimeResponseSchema
        }
      }
    }
  }, router.handleLegacyErrors((0, _handle_errors.handleErrors)(async (ctx, req, res) => {
    const core = await ctx.core;
    const savedObjectsClient = core.savedObjects.client;
    const elasticsearchClient = core.elasticsearch.client.asCurrentUser;
    const [,, {
      dataViewsServiceFactory
    }] = await getStartServices();
    const indexPatternsService = await dataViewsServiceFactory(savedObjectsClient, elasticsearchClient, req);
    const id = req.params.id;
    const name = req.params.name;
    const field = {
      ...req.body.field,
      name
    };
    const indexPattern = await indexPatternsService.getDataViewLazy(id);
    let fieldObject = await indexPattern.getFieldByName(name);
    if (!fieldObject) {
      throw new _error.ErrorIndexPatternFieldNotFound(id, name);
    }
    if (!fieldObject.scripted) {
      throw new Error('Only scripted fields can be updated.');
    }
    const oldSpec = fieldObject.toSpec();
    indexPattern.upsertScriptedField({
      ...oldSpec,
      ...field,
      name: field.name
    });
    await indexPatternsService.updateSavedObject(indexPattern);
    fieldObject = await indexPattern.getFieldByName(name);
    if (!fieldObject) throw new Error(`Could not create a field [name = ${field.name}].`);
    const body = {
      field: fieldObject.toSpec(),
      index_pattern: await indexPattern.toSpec({
        fieldParams: {
          fieldName: ['*']
        }
      })
    };
    return res.ok({
      headers: {
        'content-type': 'application/json'
      },
      body
    });
  })));
};
exports.registerUpdateScriptedFieldRoute = registerUpdateScriptedFieldRoute;