"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AbstractDataView = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _utils = require("./utils");
var _meta_units_to_formatter = require("./meta_units_to_formatter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class AbstractDataView {
  constructor(config) {
    /**
     * Saved object id
     */
    (0, _defineProperty2.default)(this, "id", void 0);
    /**
     * Title of data view
     * @deprecated use getIndexPattern instead
     */
    (0, _defineProperty2.default)(this, "title", '');
    /**
     * Map of field formats by field name
     */
    (0, _defineProperty2.default)(this, "fieldFormatMap", void 0);
    /**
     * Only used by rollup indices, used by rollup specific endpoint to load field list.
     */
    (0, _defineProperty2.default)(this, "typeMeta", void 0);
    /**
     * Timestamp field name
     */
    (0, _defineProperty2.default)(this, "timeFieldName", void 0);
    /**
     * Type is used to identify rollup index patterns or ES|QL data views.
     */
    (0, _defineProperty2.default)(this, "type", void 0);
    /**
     * List of meta fields by name
     */
    (0, _defineProperty2.default)(this, "metaFields", void 0);
    /**
     * SavedObject version
     */
    (0, _defineProperty2.default)(this, "version", void 0);
    /**
     * Array of filters - hides fields in discover
     */
    (0, _defineProperty2.default)(this, "sourceFilters", void 0);
    /**
     * Array of namespace ids
     */
    (0, _defineProperty2.default)(this, "namespaces", void 0);
    /**
     * Original saved object body. Used to check for saved object changes.
     */
    (0, _defineProperty2.default)(this, "originalSavedObjectBody", {});
    /**
     * Returns true if short dot notation is enabled
     */
    (0, _defineProperty2.default)(this, "shortDotsEnable", false);
    /**
     * FieldFormats service interface
     */
    (0, _defineProperty2.default)(this, "fieldFormats", void 0);
    /**
     * Map of field attributes by field name. Currently count and customLabel.
     */
    (0, _defineProperty2.default)(this, "fieldAttrs", void 0);
    /**
     * Map of runtime field definitions by field name
     */
    (0, _defineProperty2.default)(this, "runtimeFieldMap", void 0);
    /**
     * Prevents errors when index pattern exists before indices
     */
    (0, _defineProperty2.default)(this, "allowNoIndex", false);
    /**
     * Name of the data view. Human readable name used to differentiate data view.
     */
    (0, _defineProperty2.default)(this, "name", '');
    /*
     * list of indices that the index pattern matched
     */
    (0, _defineProperty2.default)(this, "matchedIndices", []);
    (0, _defineProperty2.default)(this, "scriptedFieldsMap", void 0);
    (0, _defineProperty2.default)(this, "allowHidden", false);
    (0, _defineProperty2.default)(this, "getAllowHidden", () => this.allowHidden);
    (0, _defineProperty2.default)(this, "setAllowHidden", allowHidden => this.allowHidden = allowHidden);
    /**
     * Get name of Data View
     */
    (0, _defineProperty2.default)(this, "getName", () => this.name ? this.name : this.title);
    /**
     * Get index pattern
     * @returns index pattern string
     */
    (0, _defineProperty2.default)(this, "getIndexPattern", () => this.title);
    /**
     * Set index pattern
     * @param string index pattern string
     */
    (0, _defineProperty2.default)(this, "setIndexPattern", indexPattern => {
      this.title = indexPattern;
    });
    /**
     * Get last saved saved object fields
     */
    (0, _defineProperty2.default)(this, "getOriginalSavedObjectBody", () => ({
      ...this.originalSavedObjectBody
    }));
    /**
     * Reset last saved saved object fields. Used after saving.
     */
    (0, _defineProperty2.default)(this, "resetOriginalSavedObjectBody", () => {
      this.originalSavedObjectBody = this.getAsSavedObjectBody();
    });
    /**
     * Set field formatter
     * @param fieldName name of field to set format on
     * @param format field format in serialized form
     */
    (0, _defineProperty2.default)(this, "setFieldFormat", (fieldName, format) => {
      this.fieldFormatMap[fieldName] = format;
    });
    /**
     * Remove field format from the field format map.
     * @param fieldName field name associated with the format for removal
     */
    (0, _defineProperty2.default)(this, "deleteFieldFormat", fieldName => {
      delete this.fieldFormatMap[fieldName];
    });
    (0, _defineProperty2.default)(this, "upsertScriptedFieldInternal", field => {
      this.scriptedFieldsMap[field.name] = {
        name: field.name,
        script: field.script,
        lang: field.lang,
        type: field.type,
        scripted: field.scripted
      };
    });
    (0, _defineProperty2.default)(this, "deleteScriptedFieldInternal", fieldName => {
      delete this.scriptedFieldsMap[fieldName];
    });
    (0, _defineProperty2.default)(this, "getFieldAttrs", () => {
      const clonedFieldAttrs = (0, _lodash.cloneDeep)(Object.fromEntries(this.fieldAttrs.entries()));
      return new Map(Object.entries(clonedFieldAttrs));
    });
    const {
      spec = {},
      fieldFormats,
      shortDotsEnable = false,
      metaFields = []
    } = config;

    // it's importing field attributes when a data view is imported from a spec and those attributes aren't provided in the fieldAttrs
    const extractedFieldAttrs = spec !== null && spec !== void 0 && spec.fields ? Object.entries(spec.fields).reduce((acc, [key, value]) => {
      const attrs = {};
      let hasAttrs = false;
      if (value.count) {
        attrs.count = value.count;
        hasAttrs = true;
      }
      if (value.customLabel) {
        attrs.customLabel = value.customLabel;
        hasAttrs = true;
      }
      if (value.customDescription) {
        attrs.customDescription = value.customDescription;
        hasAttrs = true;
      }
      if (hasAttrs) {
        acc[key] = attrs;
      }
      return acc;
    }, {}) : {};
    this.allowNoIndex = (spec === null || spec === void 0 ? void 0 : spec.allowNoIndex) || false;
    this.scriptedFieldsMap = spec !== null && spec !== void 0 && spec.fields ? Object.values(spec.fields).filter(field => field.scripted).reduce((acc, field) => {
      acc[field.name] = field;
      return acc;
    }, {}) : {};

    // set dependencies
    this.fieldFormats = {
      ...fieldFormats
    };
    // set config
    this.shortDotsEnable = shortDotsEnable;
    this.metaFields = metaFields;

    // set values
    this.id = spec.id;
    this.fieldFormatMap = {
      ...spec.fieldFormats
    };
    this.version = spec.version;
    this.title = spec.title || '';
    this.timeFieldName = spec.timeFieldName;
    this.sourceFilters = [...(spec.sourceFilters || [])];
    this.type = spec.type;
    this.typeMeta = spec.typeMeta;
    this.fieldAttrs = new Map(Object.entries({
      ...extractedFieldAttrs,
      ...spec.fieldAttrs
    }));
    this.runtimeFieldMap = (0, _lodash.cloneDeep)(spec.runtimeFieldMap) || {};
    this.namespaces = spec.namespaces || [];
    this.name = spec.name || '';
    this.allowHidden = spec.allowHidden || false;
  }
  /**
   * Returns true if the data view is persisted, and false if the dataview is adhoc.
   */
  isPersisted() {
    return typeof this.version === 'string';
  }

  /**
   * Get the source filtering configuration for that index.
   */
  getSourceFiltering() {
    return {
      excludes: this.sourceFilters && this.sourceFilters.map(filter => filter.value) || []
    };
  }

  /**
   * Get aggregation restrictions. Rollup fields can only perform a subset of aggregations.
   */

  getAggregationRestrictions() {
    var _this$typeMeta;
    return (_this$typeMeta = this.typeMeta) === null || _this$typeMeta === void 0 ? void 0 : _this$typeMeta.aggs;
  }

  /**
   * Provide a field, get its formatter
   * @param field field to get formatter for
   */
  getFormatterForField(field) {
    const fieldFormat = this.getFormatterForFieldNoDefault(field.name);
    if (fieldFormat) {
      return fieldFormat;
    }
    const fmt = field.defaultFormatter ? _meta_units_to_formatter.metaUnitsToFormatter[field.defaultFormatter] : undefined;
    if (fmt) {
      return this.fieldFormats.getInstance(fmt.id, fmt.params);
    }
    return this.fieldFormats.getDefaultInstance(field.type, field.esTypes);
  }

  /**
   * Get formatter for a given field name. Return undefined if none exists.
   * @param fieldname name of field to get formatter for
   */
  getFormatterForFieldNoDefault(fieldname) {
    const formatSpec = this.fieldFormatMap[fieldname];
    if (formatSpec !== null && formatSpec !== void 0 && formatSpec.id) {
      return this.fieldFormats.getInstance(formatSpec.id, formatSpec.params);
    }
  }

  /**
   * Set field attribute
   * @param fieldName name of field to set attribute on
   * @param attrName name of attribute to set
   * @param value value of attribute
   */

  setFieldAttrs(fieldName, attrName, value) {
    const fieldAttrs = this.fieldAttrs.get(fieldName) || {};
    this.fieldAttrs.set(fieldName, {
      ...fieldAttrs,
      [attrName]: value
    });
  }

  /**
   * Set field custom label
   * @param fieldName name of field to set custom label on
   * @param customLabel custom label value. If undefined, custom label is removed
   */

  setFieldCustomLabelInternal(fieldName, customLabel) {
    this.setFieldAttrs(fieldName, 'customLabel', customLabel === null ? undefined : customLabel);
  }

  /**
   * Set field count
   * @param fieldName name of field to set count on
   * @param count count value. If undefined, count is removed
   */

  setFieldCountInternal(fieldName, count) {
    this.setFieldAttrs(fieldName, 'count', count === null ? undefined : count);
  }

  /**
   * Set field custom description
   * @param fieldName name of field to set custom description on
   * @param customDescription custom description value. If undefined, custom description is removed
   */

  setFieldCustomDescriptionInternal(fieldName, customDescription) {
    this.setFieldAttrs(fieldName, 'customDescription', customDescription === null ? undefined : customDescription);
  }
  /**
   * Returns index pattern as saved object body for saving
   */
  getAsSavedObjectBody() {
    const stringifyOrUndefined = obj => obj ? JSON.stringify(obj) : undefined;
    const fieldAttrsWithValues = {};
    this.fieldAttrs.forEach((attrs, fieldName) => {
      if (Object.keys(attrs).length) {
        fieldAttrsWithValues[fieldName] = attrs;
      }
    });
    return {
      fieldAttrs: stringifyOrUndefined(fieldAttrsWithValues),
      title: this.getIndexPattern(),
      timeFieldName: this.timeFieldName,
      sourceFilters: stringifyOrUndefined(this.sourceFilters),
      fields: stringifyOrUndefined(Object.values(this.scriptedFieldsMap)),
      fieldFormatMap: stringifyOrUndefined(this.fieldFormatMap),
      type: this.type,
      typeMeta: stringifyOrUndefined(this.typeMeta),
      allowNoIndex: this.allowNoIndex ? this.allowNoIndex : undefined,
      runtimeFieldMap: stringifyOrUndefined(this.runtimeFieldMap),
      name: this.name,
      allowHidden: this.allowHidden
    };
  }
  toSpecShared(includeFields = true) {
    // `cloneDeep` is added to make sure that the original fieldAttrs map is not modified with the following `delete` operation.
    const fieldAttrs = (0, _lodash.cloneDeep)(Object.fromEntries(this.fieldAttrs.entries()));

    // if fields aren't included, don't include count
    if (!includeFields) {
      Object.keys(fieldAttrs).forEach(key => {
        delete fieldAttrs[key].count;
        if (Object.keys(fieldAttrs[key]).length === 0) {
          delete fieldAttrs[key];
        }
      });
    }
    const spec = {
      id: this.id,
      version: this.version,
      title: this.getIndexPattern(),
      timeFieldName: this.timeFieldName,
      sourceFilters: [...(this.sourceFilters || [])],
      typeMeta: this.typeMeta,
      type: this.type,
      fieldFormats: {
        ...this.fieldFormatMap
      },
      runtimeFieldMap: (0, _lodash.cloneDeep)(this.runtimeFieldMap),
      fieldAttrs,
      allowNoIndex: this.allowNoIndex,
      name: this.name,
      allowHidden: this.getAllowHidden()
    };

    // Filter undefined values from the spec
    return Object.fromEntries(Object.entries(spec).filter(([, v]) => typeof v !== 'undefined'));
  }
  replaceAllScriptedFields(newFields) {
    const oldScriptedFieldNames = Object.keys(this.scriptedFieldsMap);
    oldScriptedFieldNames.forEach(name => {
      this.removeScriptedField(name);
    });
    Object.entries(newFields).forEach(([name, field]) => {
      this.upsertScriptedField(field);
    });
  }
  removeScriptedField(name) {
    return this.deleteScriptedFieldInternal(name);
  }
  upsertScriptedField(field) {
    return this.upsertScriptedFieldInternal(field);
  }

  /**
   * Only used by search source to process sorting of scripted fields
   * @param name field name
   * @returns DataViewFieldBase
   */
  getScriptedField(name) {
    // runtime fields override scripted fields
    if (this.runtimeFieldMap[name]) {
      return;
    }
    const field = this.scriptedFieldsMap[name];
    if (field) {
      return {
        ...field,
        scripted: true
      };
    }
  }

  /**
   * Checks if runtime field exists
   * @param name field name
   */
  hasRuntimeField(name) {
    return !!this.runtimeFieldMap[name];
  }

  /**
   * Returns runtime field if exists
   * @param name Runtime field name
   */
  getRuntimeField(name) {
    if (!this.runtimeFieldMap[name]) {
      return null;
    }
    const {
      type,
      script,
      fields
    } = {
      ...this.runtimeFieldMap[name]
    };
    const runtimeField = {
      type,
      script
    };
    if (type === 'composite') {
      runtimeField.fields = fields;
    }
    return runtimeField;
  }

  /**
   * Get all runtime field definitions.
   * NOTE: this does not strip out runtime fields that match mapped field names
   * @returns map of runtime field definitions by field name
   */

  getAllRuntimeFields() {
    return Object.keys(this.runtimeFieldMap).reduce((acc, fieldName) => ({
      ...acc,
      [fieldName]: this.getRuntimeField(fieldName)
    }), {});
  }

  /**
   * Replaces all existing runtime fields with new fields.
   * @param newFields Map of runtime field definitions by field name
   */
  replaceAllRuntimeFields(newFields) {
    const oldRuntimeFieldNames = Object.keys(this.runtimeFieldMap);
    oldRuntimeFieldNames.forEach(name => {
      this.removeRuntimeField(name);
    });
    Object.entries(newFields).forEach(([name, field]) => {
      this.addRuntimeField(name, field);
    });
  }
  removeRuntimeField(name) {
    return this.removeRuntimeFieldInteral(name);
  }
  addRuntimeField(name, runtimeField) {
    return this.addRuntimeFieldInteral(name, runtimeField);
  }
  removeRuntimeFieldInteral(name) {
    delete this.runtimeFieldMap[name];
  }
  addRuntimeFieldInteral(name, runtimeField) {
    this.runtimeFieldMap[name] = (0, _utils.removeFieldAttrs)(runtimeField);
  }
  /**
   * Checks if there are any matched indices.
   * @returns True if there are matched indices, false otherwise.
   */
  hasMatchedIndices() {
    return !!this.matchedIndices.length;
  }
}
exports.AbstractDataView = AbstractDataView;