"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDataStream = createDataStream;
var _nunjucks = _interopRequireDefault(require("nunjucks"));
var _path = require("path");
var _jsYaml = require("js-yaml");
var _constants = require("./constants");
var _util = require("../util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createDataStream(packageName, specificDataStreamDir, dataStream) {
  const dataStreamName = dataStream.name;
  const pipelineDir = (0, _path.join)(specificDataStreamDir, 'elasticsearch', 'ingest_pipeline');
  const title = dataStream.title;
  const description = dataStream.description;
  const samplesFormat = dataStream.samplesFormat;
  const useMultilineNDJSON = samplesFormat.name === 'ndjson' && samplesFormat.multiline === true;
  (0, _util.ensureDirSync)(specificDataStreamDir);
  const fields = createDataStreamFolders(specificDataStreamDir, pipelineDir);
  const dataStreams = [];
  for (const inputType of dataStream.inputTypes) {
    let mappedValues = {
      data_stream_title: title,
      data_stream_description: description,
      package_name: packageName,
      data_stream_name: dataStreamName,
      multiline_ndjson: useMultilineNDJSON
    };
    if (inputType === 'cel') {
      mappedValues = prepareCelValues(mappedValues, dataStream.celInput);
    }
    const dataStreamManifest = _nunjucks.default.render(`${inputType.replaceAll('-', '_')}_manifest.yml.njk`, mappedValues);
    const commonManifest = _nunjucks.default.render('common_manifest.yml.njk', mappedValues);
    const combinedManifest = `${dataStreamManifest}\n${commonManifest}`;
    dataStreams.push(combinedManifest);
  }
  const finalManifest = _nunjucks.default.render('data_stream.yml.njk', {
    title,
    data_streams: dataStreams
  });
  (0, _util.createSync)((0, _path.join)(specificDataStreamDir, 'manifest.yml'), finalManifest);
  return fields;
}
function createDataStreamFolders(specificDataStreamDir, pipelineDir) {
  (0, _util.ensureDirSync)(pipelineDir);
  return copyFilesFromTemplateDir(specificDataStreamDir);
}
function copyFilesFromTemplateDir(specificDataStreamDir) {
  const dataStreamTemplatesDir = (0, _path.join)(__dirname, '../templates/data_stream');
  const items = (0, _util.listDirSync)(dataStreamTemplatesDir);
  return items.flatMap(item => {
    const sourcePath = (0, _path.join)(dataStreamTemplatesDir, item);
    const destinationPath = (0, _path.join)(specificDataStreamDir, item);
    (0, _util.copySync)(sourcePath, destinationPath);
    const files = (0, _util.listDirSync)(sourcePath);
    return loadFieldsFromFiles(sourcePath, files);
  });
}
function loadFieldsFromFiles(sourcePath, files) {
  return files.flatMap(file => {
    const filePath = (0, _path.join)(sourcePath, file);
    const content = (0, _util.readSync)(filePath);
    return (0, _jsYaml.load)(content);
  });
}
function prepareCelValues(mappedValues, celInput) {
  if (celInput != null) {
    // Ready the program for printing with correct indentation
    const programLines = celInput.program.split('\n');
    // We don't want to double include the config fields in the state or any of the templated auth fields
    const initialState = Object.entries(celInput.stateSettings).filter(([key]) => !Object.keys(celInput.configFields).includes(key) && !_constants.CEL_EXISTING_AUTH_CONFIG_FIELDS.includes(key));
    const configSettingsNeeded = Object.entries(celInput.configFields).filter(([key]) => !_constants.CEL_EXISTING_AUTH_CONFIG_FIELDS.includes(key));
    return {
      ...mappedValues,
      program: programLines,
      state: initialState,
      configFields: configSettingsNeeded,
      redact: celInput.redactVars,
      auth: celInput.authType,
      url: celInput.url,
      showAll: false
    };
  } else {
    return {
      ...mappedValues,
      program: _constants.DEFAULT_CEL_PROGRAM.split('\n'),
      url: _constants.DEFAULT_URL,
      showAll: true
    };
  }
}