/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.core.ssl;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.common.ssl.SslConfiguration;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.FileWatcher;
import org.elasticsearch.watcher.ResourceWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;
import org.elasticsearch.xpack.core.ssl.SSLService;

public final class SSLConfigurationReloader {
    private static final Logger logger = LogManager.getLogger(SSLConfigurationReloader.class);
    private final PlainActionFuture<SSLService> sslServiceFuture = new PlainActionFuture<SSLService>(this){

        protected boolean blockingAllowed() {
            return true;
        }
    };

    public SSLConfigurationReloader(ResourceWatcherService resourceWatcherService, Collection<SslConfiguration> sslConfigurations) {
        SSLConfigurationReloader.startWatching(SSLConfigurationReloader.reloadConsumer(this.sslServiceFuture), resourceWatcherService, sslConfigurations);
    }

    SSLConfigurationReloader(Consumer<SslConfiguration> reloadConsumer, ResourceWatcherService resourceWatcherService, Collection<SslConfiguration> sslConfigurations) {
        SSLConfigurationReloader.startWatching(reloadConsumer, resourceWatcherService, sslConfigurations);
    }

    public void setSSLService(SSLService sslService) {
        assert (!this.sslServiceFuture.isDone()) : "ssl service future was already completed!";
        this.sslServiceFuture.onResponse((Object)sslService);
    }

    private static Consumer<SslConfiguration> reloadConsumer(Future<SSLService> future) {
        return sslConfiguration -> {
            try {
                SSLService sslService = (SSLService)future.get();
                logger.debug("reloading ssl configuration [{}]", sslConfiguration);
                sslService.reloadSSLContext((SslConfiguration)sslConfiguration);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (ExecutionException e) {
                throw new ElasticsearchException("failed to obtain ssl service", (Throwable)e, new Object[0]);
            }
        };
    }

    private static void startWatching(Consumer<SslConfiguration> reloadConsumer, ResourceWatcherService resourceWatcherService, Collection<SslConfiguration> sslConfigurations) {
        HashMap<Path, List> pathToConfigurationsMap = new HashMap<Path, List>();
        for (SslConfiguration sslConfiguration : sslConfigurations) {
            Collection filesToMonitor = sslConfiguration.getDependentFiles();
            for (Path file : filesToMonitor) {
                pathToConfigurationsMap.compute(file, (path, list) -> {
                    if (list == null) {
                        list = new ArrayList<SslConfiguration>();
                    }
                    list.add(sslConfiguration);
                    return list;
                });
            }
        }
        pathToConfigurationsMap.forEach((path, configurations) -> {
            ChangeListener changeListener = new ChangeListener(List.copyOf(configurations), reloadConsumer);
            FileWatcher fileWatcher = new FileWatcher(path);
            fileWatcher.addListener((Object)changeListener);
            try {
                resourceWatcherService.add((ResourceWatcher)fileWatcher, ResourceWatcherService.Frequency.HIGH);
            }
            catch (IOException | SecurityException e) {
                logger.error("failed to start watching file [{}] for ssl configurations [{}] - {}", path, configurations, (Object)e);
            }
        });
    }

    private static class ChangeListener
    implements FileChangesListener {
        private final List<SslConfiguration> sslConfigurations;
        private final Consumer<SslConfiguration> reloadConsumer;

        private ChangeListener(List<SslConfiguration> sslConfigurations, Consumer<SslConfiguration> reloadConsumer) {
            this.sslConfigurations = sslConfigurations;
            this.reloadConsumer = reloadConsumer;
        }

        public void onFileCreated(Path file) {
            this.onFileChanged(file);
        }

        public void onFileDeleted(Path file) {
            this.onFileChanged(file);
        }

        public void onFileChanged(Path file) {
            long reloadedNanos = System.nanoTime();
            ArrayList<String> settingPrefixes = new ArrayList<String>(this.sslConfigurations.size());
            for (SslConfiguration sslConfiguration : this.sslConfigurations) {
                if (!sslConfiguration.getDependentFiles().contains(file)) continue;
                this.reloadConsumer.accept(sslConfiguration);
                settingPrefixes.add(sslConfiguration.settingPrefix());
            }
            if (!settingPrefixes.isEmpty()) {
                logger.info("updated {} ssl contexts in {}ms for prefix names {} using file [{}]", (Object)settingPrefixes.size(), (Object)TimeValue.timeValueNanos((long)(System.nanoTime() - reloadedNanos)).millisFrac(), settingPrefixes, (Object)file);
            }
        }
    }
}

