/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.collect;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.BiPredicate;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.IntFunction;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.function.ToIntFunction;
import org.elasticsearch.core.Nullable;

public class Iterators {
    public static <T> Iterator<T> single(T element) {
        return new SingleIterator<T>(element);
    }

    @SafeVarargs
    public static <T> Iterator<T> concat(Iterator<? extends T> ... iterators) {
        if (iterators == null) {
            throw new NullPointerException("iterators");
        }
        for (int i = 0; i < iterators.length; ++i) {
            if (!iterators[i].hasNext()) continue;
            return new ConcatenatedIterator<T>(iterators, i);
        }
        return Collections.emptyIterator();
    }

    public static <T> Iterator<T> forArray(T[] array) {
        return Arrays.asList(array).iterator();
    }

    public static <T> Iterator<T> forRange(int lowerBoundInclusive, int upperBoundExclusive, IntFunction<? extends T> fn) {
        assert (lowerBoundInclusive <= upperBoundExclusive) : lowerBoundInclusive + " vs " + upperBoundExclusive;
        if (upperBoundExclusive <= lowerBoundInclusive) {
            return Collections.emptyIterator();
        }
        return new IntRangeIterator<T>(lowerBoundInclusive, upperBoundExclusive, Objects.requireNonNull(fn));
    }

    public static <T, U> Iterator<U> map(Iterator<? extends T> input, Function<T, ? extends U> fn) {
        if (input.hasNext()) {
            if (input instanceof MapIterator) {
                MapIterator mapIterator = (MapIterator)input;
                return new MapIterator(mapIterator.input, mapIterator.fn.andThen(fn));
            }
            return new MapIterator<T, U>(input, fn);
        }
        return Collections.emptyIterator();
    }

    public static <T> Iterator<T> filter(Iterator<? extends T> input, Predicate<T> predicate) {
        while (input.hasNext()) {
            T value = input.next();
            assert (value != null);
            if (!predicate.test(value)) continue;
            return new FilterIterator<T>(value, input, predicate);
        }
        return Collections.emptyIterator();
    }

    public static <T> Iterator<T> limit(Iterator<? extends T> input, int n) {
        assert (n >= 0) : "negative limit";
        if (n > 0 && input.hasNext()) {
            return new LimitIterator<T>(input, n);
        }
        return Collections.emptyIterator();
    }

    public static <T> List<T> toList(Iterator<T> iterator) {
        if (iterator.hasNext()) {
            ArrayList<T> list = new ArrayList<T>();
            while (iterator.hasNext()) {
                list.add(iterator.next());
            }
            return Collections.unmodifiableList(list);
        }
        return Collections.emptyList();
    }

    public static <T, U> Iterator<U> flatMap(Iterator<? extends T> input, Function<T, Iterator<? extends U>> fn) {
        while (input.hasNext()) {
            Iterator<U> value = fn.apply(input.next());
            if (!value.hasNext()) continue;
            return new FlatMapIterator<T, U>(input, fn, value);
        }
        return Collections.emptyIterator();
    }

    public static <T> Iterator<T> failFast(Iterator<T> input, BooleanSupplier isFailingSupplier) {
        if (isFailingSupplier.getAsBoolean()) {
            return Collections.emptyIterator();
        }
        return new FailFastIterator<T>(input, isFailingSupplier);
    }

    public static <T, U> Iterator<U> enumerate(Iterator<? extends T> input, BiFunction<Integer, T, ? extends U> fn) {
        return new EnumeratingIterator<T, U>(Objects.requireNonNull(input), Objects.requireNonNull(fn));
    }

    public static <T> Iterator<T> fromSupplier(Supplier<? extends T> input) {
        return new SupplierIterator<T>(Objects.requireNonNull(input));
    }

    public static <T> Iterator<T> cycling(Iterable<T> source) {
        return new CyclingIterator<T>(Objects.requireNonNull(source));
    }

    public static <T> boolean equals(Iterator<? extends T> iterator1, Iterator<? extends T> iterator2, BiPredicate<T, T> itemComparer) {
        if (iterator1 == null) {
            return iterator2 == null;
        }
        if (iterator2 == null) {
            return false;
        }
        while (iterator1.hasNext()) {
            if (!iterator2.hasNext()) {
                return false;
            }
            if (itemComparer.test(iterator1.next(), iterator2.next())) continue;
            return false;
        }
        return !iterator2.hasNext();
    }

    public static <T> int hashCode(Iterator<? extends T> iterator, ToIntFunction<T> itemHashcode) {
        if (iterator == null) {
            return 0;
        }
        int result = 1;
        while (iterator.hasNext()) {
            result = 31 * result + itemHashcode.applyAsInt(iterator.next());
        }
        return result;
    }

    private static final class SingleIterator<T>
    implements Iterator<T> {
        private T value;

        SingleIterator(T element) {
            this.value = Objects.requireNonNull(element);
        }

        @Override
        public boolean hasNext() {
            return this.value != null;
        }

        @Override
        public T next() {
            T res = this.value;
            this.value = null;
            return res;
        }
    }

    private static class ConcatenatedIterator<T>
    implements Iterator<T> {
        private final Iterator<? extends T>[] iterators;
        private int index;

        ConcatenatedIterator(Iterator<? extends T>[] iterators, int startIndex) {
            for (int i = startIndex; i < iterators.length; ++i) {
                if (iterators[i] != null) continue;
                throw new NullPointerException("iterators[" + i + "]");
            }
            this.iterators = iterators;
            this.index = startIndex;
        }

        @Override
        public boolean hasNext() {
            return this.index < this.iterators.length;
        }

        @Override
        public T next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            T value = this.iterators[this.index].next();
            while (this.index < this.iterators.length && !this.iterators[this.index].hasNext()) {
                ++this.index;
            }
            return value;
        }

        @Override
        public void forEachRemaining(Consumer<? super T> action) {
            while (this.index < this.iterators.length) {
                this.iterators[this.index++].forEachRemaining(action);
            }
        }
    }

    private static final class IntRangeIterator<T>
    implements Iterator<T> {
        private final IntFunction<? extends T> fn;
        private final int upperBoundExclusive;
        private int index;

        IntRangeIterator(int lowerBoundInclusive, int upperBoundExclusive, IntFunction<? extends T> fn) {
            this.fn = fn;
            this.index = lowerBoundInclusive;
            this.upperBoundExclusive = upperBoundExclusive;
        }

        @Override
        public boolean hasNext() {
            return this.index < this.upperBoundExclusive;
        }

        @Override
        public T next() {
            if (this.index >= this.upperBoundExclusive) {
                throw new NoSuchElementException();
            }
            return this.fn.apply(this.index++);
        }
    }

    private static final class MapIterator<T, U>
    implements Iterator<U> {
        private final Iterator<? extends T> input;
        private final Function<T, ? extends U> fn;

        MapIterator(Iterator<? extends T> input, Function<T, ? extends U> fn) {
            this.input = input;
            this.fn = fn;
        }

        @Override
        public boolean hasNext() {
            return this.input.hasNext();
        }

        @Override
        public U next() {
            return this.fn.apply(this.input.next());
        }

        @Override
        public void forEachRemaining(Consumer<? super U> action) {
            this.input.forEachRemaining((? super E t) -> action.accept((U)this.fn.apply(t)));
        }
    }

    private static final class FilterIterator<T>
    implements Iterator<T> {
        private final Iterator<? extends T> input;
        private final Predicate<T> predicate;
        private T next;

        FilterIterator(T value, Iterator<? extends T> input, Predicate<T> predicate) {
            this.next = value;
            this.input = input;
            this.predicate = predicate;
            assert (this.next != null);
            assert (predicate.test(this.next));
        }

        @Override
        public boolean hasNext() {
            return this.next != null;
        }

        @Override
        public T next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            T value = this.next;
            while (this.input.hasNext()) {
                T laterValue = this.input.next();
                assert (laterValue != null);
                if (!this.predicate.test(laterValue)) continue;
                this.next = laterValue;
                return value;
            }
            this.next = null;
            return value;
        }
    }

    private static final class LimitIterator<T>
    implements Iterator<T> {
        private final Iterator<? extends T> input;
        private final int limit;
        private int current;

        LimitIterator(Iterator<? extends T> input, int limit) {
            this.input = input;
            this.limit = limit;
        }

        @Override
        public boolean hasNext() {
            return this.current < this.limit && this.input.hasNext();
        }

        @Override
        public T next() {
            if (this.current >= this.limit) {
                throw new NoSuchElementException();
            }
            ++this.current;
            return this.input.next();
        }
    }

    private static final class FlatMapIterator<T, U>
    implements Iterator<U> {
        private final Iterator<? extends T> input;
        private final Function<T, Iterator<? extends U>> fn;
        @Nullable
        private Iterator<? extends U> currentOutput;

        FlatMapIterator(Iterator<? extends T> input, Function<T, Iterator<? extends U>> fn, Iterator<? extends U> firstOutput) {
            this.input = input;
            this.fn = fn;
            this.currentOutput = firstOutput;
        }

        @Override
        public boolean hasNext() {
            return this.currentOutput != null;
        }

        @Override
        public U next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            assert (this.currentOutput != null && this.currentOutput.hasNext());
            U value = this.currentOutput.next();
            while (this.currentOutput != null && !this.currentOutput.hasNext()) {
                if (this.input.hasNext()) {
                    this.currentOutput = this.fn.apply(this.input.next());
                    continue;
                }
                this.currentOutput = null;
            }
            return value;
        }
    }

    private static class FailFastIterator<T>
    implements Iterator<T> {
        private final Iterator<T> delegate;
        private final BooleanSupplier isFailingSupplier;

        FailFastIterator(Iterator<T> delegate, BooleanSupplier isFailingSupplier) {
            this.delegate = delegate;
            this.isFailingSupplier = isFailingSupplier;
        }

        @Override
        public boolean hasNext() {
            return !this.isFailingSupplier.getAsBoolean() && this.delegate.hasNext();
        }

        @Override
        public T next() {
            return this.delegate.next();
        }
    }

    private static class EnumeratingIterator<T, U>
    implements Iterator<U> {
        private final Iterator<? extends T> input;
        private final BiFunction<Integer, T, ? extends U> fn;
        private int idx = 0;

        EnumeratingIterator(Iterator<? extends T> input, BiFunction<Integer, T, ? extends U> fn) {
            this.input = input;
            this.fn = fn;
        }

        @Override
        public boolean hasNext() {
            return this.input.hasNext();
        }

        @Override
        public U next() {
            return this.fn.apply(this.idx++, (Integer)this.input.next());
        }

        @Override
        public void forEachRemaining(Consumer<? super U> action) {
            this.input.forEachRemaining((? super E t) -> action.accept((U)this.fn.apply(this.idx++, (Integer)t)));
        }
    }

    private static final class SupplierIterator<T>
    implements Iterator<T> {
        private final Supplier<? extends T> fn;
        private T head;

        SupplierIterator(Supplier<? extends T> fn) {
            this.fn = fn;
            this.head = fn.get();
        }

        @Override
        public boolean hasNext() {
            return this.head != null;
        }

        @Override
        public T next() {
            if (this.head == null) {
                throw new NoSuchElementException();
            }
            T next = this.head;
            this.head = this.fn.get();
            return next;
        }
    }

    private static class CyclingIterator<T>
    implements Iterator<T> {
        private final Iterable<T> source;
        private Iterator<T> iterator;

        CyclingIterator(Iterable<T> source) {
            this.source = source;
            this.iterator = source.iterator();
            if (!this.iterator.hasNext()) {
                throw new IllegalArgumentException("Cannot cycle over empty iterable");
            }
        }

        @Override
        public boolean hasNext() {
            return true;
        }

        @Override
        public T next() {
            if (this.iterator.hasNext()) {
                return this.iterator.next();
            }
            this.iterator = this.source.iterator();
            if (!this.iterator.hasNext()) {
                throw new IllegalArgumentException("Cannot cycle over empty iterable");
            }
            return this.iterator.next();
        }
    }
}

