/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.bytes;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Objects;
import java.util.stream.IntStream;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.BytesRefIterator;
import org.apache.lucene.util.RamUsageEstimator;
import org.elasticsearch.common.bytes.AbstractBytesReference;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;

public final class CompositeBytesReference
extends AbstractBytesReference {
    private final BytesReference[] references;
    private final int[] offsets;
    private final long ramBytesUsed;

    public static BytesReference of(BytesReference ... references) {
        if (references.length == 0) {
            return BytesArray.EMPTY;
        }
        if (references.length == 1) {
            return references[0];
        }
        return CompositeBytesReference.ofMultiple(references);
    }

    private static BytesReference ofMultiple(BytesReference[] references) {
        assert (references.length > 1) : "use #of() instead";
        int[] offsets = new int[references.length];
        long ramBytesUsed = 0L;
        int offset = 0;
        for (int i = 0; i < references.length; ++i) {
            BytesReference reference = references[i];
            if (reference == null) {
                throw new IllegalArgumentException("references must not be null");
            }
            if (reference.length() == 0) {
                return CompositeBytesReference.dropEmptyReferences(references);
            }
            offsets[i] = offset;
            if ((offset += reference.length()) <= 0) {
                throw new IllegalArgumentException("CompositeBytesReference cannot hold more than 2GB");
            }
            ramBytesUsed += reference.ramBytesUsed();
        }
        return new CompositeBytesReference(references, offsets, offset, ramBytesUsed + (long)(4 * offsets.length + RamUsageEstimator.NUM_BYTES_ARRAY_HEADER) + (long)(references.length * RamUsageEstimator.NUM_BYTES_OBJECT_REF + RamUsageEstimator.NUM_BYTES_ARRAY_HEADER) + 4L + 8L);
    }

    private static BytesReference dropEmptyReferences(BytesReference[] references) {
        BytesReference[] tempArray = new BytesReference[references.length];
        int targetIndex = 0;
        for (BytesReference reference : references) {
            if (reference.length() == 0) continue;
            tempArray[targetIndex++] = reference;
        }
        assert (targetIndex < references.length) : "no empty references found";
        BytesReference[] filteredReferences = new BytesReference[targetIndex];
        System.arraycopy(tempArray, 0, filteredReferences, 0, targetIndex);
        return CompositeBytesReference.of(filteredReferences);
    }

    private CompositeBytesReference(BytesReference[] references, int[] offsets, int length, long ramBytesUsed) {
        super(length);
        assert (references != null && offsets != null);
        assert (references.length > 1) : "Should not build composite reference from less than two references but received [" + references.length + "]";
        assert (Arrays.stream(references).allMatch(r -> r != null && r.length() > 0));
        assert (offsets[0] == 0);
        assert (IntStream.range(1, references.length).allMatch(i -> offsets[i] - offsets[i - 1] == references[i - 1].length()));
        assert ((long)length == Arrays.stream(references).mapToLong(BytesReference::length).sum());
        assert (ramBytesUsed > Arrays.stream(references).mapToLong(BytesReference::ramBytesUsed).sum());
        this.references = Objects.requireNonNull(references, "references must not be null");
        this.offsets = offsets;
        this.ramBytesUsed = ramBytesUsed;
    }

    @Override
    public byte get(int index) {
        int i = this.getOffsetIndex(index);
        return this.references[i].get(index - this.offsets[i]);
    }

    @Override
    public int indexOf(byte marker, int from) {
        int remainingBytes = Math.max(this.length - from, 0);
        Objects.checkFromIndexSize(from, remainingBytes, this.length);
        int result = -1;
        if (this.length == 0) {
            return result;
        }
        int firstReferenceIndex = this.getOffsetIndex(from);
        BytesReference[] referencesAsLocal = this.references;
        int[] offsetsAsLocal = this.offsets;
        for (int i = firstReferenceIndex; i < referencesAsLocal.length; ++i) {
            BytesReference reference = referencesAsLocal[i];
            int internalFrom = i == firstReferenceIndex ? from - offsetsAsLocal[firstReferenceIndex] : 0;
            result = reference.indexOf(marker, internalFrom);
            if (result == -1) continue;
            result += offsetsAsLocal[i];
            break;
        }
        return result;
    }

    @Override
    public BytesReference slice(int from, int length) {
        int start;
        if (from == 0 && this.length == length) {
            return this;
        }
        Objects.checkFromIndexSize(from, length, this.length);
        if (length == 0) {
            return BytesArray.EMPTY;
        }
        int to = from + length;
        int limit = start = this.getOffsetIndex(from);
        int i = start + 1;
        while (i < this.offsets.length && this.offsets[i] < to) {
            limit = i++;
        }
        int inSliceOffset = from - this.offsets[start];
        if (start == limit) {
            return this.references[start].slice(inSliceOffset, length);
        }
        BytesReference[] inSlice = new BytesReference[1 + (limit - start)];
        System.arraycopy(this.references, start, inSlice, 0, inSlice.length);
        inSlice[0] = inSlice[0].slice(inSliceOffset, inSlice[0].length() - inSliceOffset);
        inSlice[inSlice.length - 1] = inSlice[inSlice.length - 1].slice(0, to - this.offsets[limit]);
        return CompositeBytesReference.ofMultiple(inSlice);
    }

    private int getOffsetIndex(int offset) {
        int i = Arrays.binarySearch(this.offsets, offset);
        return i < 0 ? -(i + 1) - 1 : i;
    }

    @Override
    public BytesRef toBytesRef() {
        byte[] result = new byte[this.length];
        int offset = 0;
        for (BytesReference reference : this.references) {
            if (reference.hasArray()) {
                int len = reference.length();
                System.arraycopy(reference.array(), reference.arrayOffset(), result, offset, len);
                offset += len;
                continue;
            }
            offset = CompositeBytesReference.copyViaIterator(reference, result, offset);
        }
        assert (offset == result.length);
        return new BytesRef(result);
    }

    private static int copyViaIterator(BytesReference reference, byte[] result, int offset) {
        BytesRefIterator iterator = reference.iterator();
        try {
            BytesRef spare;
            while ((spare = iterator.next()) != null) {
                System.arraycopy(spare.bytes, spare.offset, result, offset, spare.length);
                offset += spare.length;
            }
        }
        catch (IOException ex) {
            throw new AssertionError("won't happen", ex);
        }
        return offset;
    }

    @Override
    public BytesRefIterator iterator() {
        return new BytesRefIterator(){
            int index = 0;
            private BytesRefIterator current;
            {
                this.current = CompositeBytesReference.this.references[this.index++].iterator();
            }

            @Override
            public BytesRef next() throws IOException {
                BytesRef next = this.current.next();
                if (next == null) {
                    while (this.index < CompositeBytesReference.this.references.length) {
                        this.current = CompositeBytesReference.this.references[this.index++].iterator();
                        next = this.current.next();
                        if (next == null) continue;
                        break;
                    }
                }
                return next;
            }
        };
    }

    @Override
    public void writeTo(OutputStream os) throws IOException {
        for (BytesReference reference : this.references) {
            reference.writeTo(os);
        }
    }

    @Override
    public long ramBytesUsed() {
        return this.ramBytesUsed;
    }

    @Override
    public int getIntLE(int index) {
        BytesReference wholeIntLivesHere;
        int i = this.getOffsetIndex(index);
        int idx = index - this.offsets[i];
        int end = idx + 4;
        if (end <= (wholeIntLivesHere = this.references[i]).length()) {
            return wholeIntLivesHere.getIntLE(idx);
        }
        return super.getIntLE(index);
    }

    @Override
    public long getLongLE(int index) {
        BytesReference wholeLongsLivesHere;
        int i = this.getOffsetIndex(index);
        int idx = index - this.offsets[i];
        int end = idx + 8;
        if (end <= (wholeLongsLivesHere = this.references[i]).length()) {
            return wholeLongsLivesHere.getLongLE(idx);
        }
        return super.getLongLE(index);
    }

    @Override
    public double getDoubleLE(int index) {
        BytesReference wholeDoublesLivesHere;
        int i = this.getOffsetIndex(index);
        int idx = index - this.offsets[i];
        int end = idx + 8;
        if (end <= (wholeDoublesLivesHere = this.references[i]).length()) {
            return wholeDoublesLivesHere.getDoubleLE(idx);
        }
        return super.getDoubleLE(index);
    }
}

