/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.search.suggest.completion.context;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import org.apache.lucene.document.LatLonDocValuesField;
import org.apache.lucene.document.LatLonPoint;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.IndexableField;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.NumericUtils;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.common.geo.GeoPoint;
import org.elasticsearch.common.geo.GeoUtils;
import org.elasticsearch.common.logging.DeprecationCategory;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.unit.DistanceUnit;
import org.elasticsearch.geometry.utils.Geohash;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.index.mapper.DocumentParserContext;
import org.elasticsearch.index.mapper.GeoPointFieldMapper;
import org.elasticsearch.index.mapper.LuceneDocument;
import org.elasticsearch.index.mapper.MappedFieldType;
import org.elasticsearch.search.suggest.completion.context.ContextBuilder;
import org.elasticsearch.search.suggest.completion.context.ContextMapping;
import org.elasticsearch.search.suggest.completion.context.GeoQueryContext;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentParser;

public class GeoContextMapping
extends ContextMapping<GeoQueryContext> {
    public static final String FIELD_PRECISION = "precision";
    public static final String FIELD_FIELDNAME = "path";
    public static final int DEFAULT_PRECISION = 6;
    static final String CONTEXT_VALUE = "context";
    static final String CONTEXT_BOOST = "boost";
    static final String CONTEXT_PRECISION = "precision";
    static final String CONTEXT_NEIGHBOURS = "neighbours";
    private static final DeprecationLogger deprecationLogger = DeprecationLogger.getLogger(GeoContextMapping.class);
    private final int precision;
    private final String fieldName;

    private GeoContextMapping(String name, String fieldName, int precision) {
        super(ContextMapping.Type.GEO, name);
        this.precision = precision;
        this.fieldName = fieldName;
    }

    public String getFieldName() {
        return this.fieldName;
    }

    public int getPrecision() {
        return this.precision;
    }

    protected static GeoContextMapping load(String name, Map<String, Object> config) {
        Builder builder = new Builder(name);
        if (config != null) {
            Object fieldName;
            Object configPrecision = config.get("precision");
            if (configPrecision != null) {
                if (configPrecision instanceof Integer) {
                    builder.precision((Integer)configPrecision);
                } else if (configPrecision instanceof Long) {
                    builder.precision(((Long)configPrecision).longValue());
                } else if (configPrecision instanceof Double) {
                    builder.precision((Double)configPrecision);
                } else if (configPrecision instanceof Float) {
                    builder.precision(((Float)configPrecision).floatValue());
                } else {
                    builder.precision(configPrecision.toString());
                }
                config.remove("precision");
            }
            if ((fieldName = config.get(FIELD_FIELDNAME)) != null) {
                builder.field(fieldName.toString());
                config.remove(FIELD_FIELDNAME);
            }
        }
        return builder.build();
    }

    @Override
    protected XContentBuilder toInnerXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
        builder.field("precision", this.precision);
        if (this.fieldName != null) {
            builder.field(FIELD_FIELDNAME, this.fieldName);
        }
        return builder;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public Set<String> parseContext(DocumentParserContext documentParserContext, XContentParser parser) throws IOException, ElasticsearchParseException {
        HashSet<String> contexts = new HashSet<String>();
        XContentParser.Token token = parser.currentToken();
        if (token == XContentParser.Token.START_ARRAY) {
            token = parser.nextToken();
            if (token == XContentParser.Token.VALUE_NUMBER) {
                double lon = parser.doubleValue();
                if (parser.nextToken() != XContentParser.Token.VALUE_NUMBER) throw new ElasticsearchParseException("latitude must be a numeric value", new Object[0]);
                double lat = parser.doubleValue();
                if (parser.nextToken() != XContentParser.Token.END_ARRAY) throw new ElasticsearchParseException("only two values [lon, lat] expected", new Object[0]);
                contexts.add(Geohash.stringEncode(lon, lat, this.precision));
                return contexts;
            } else {
                while (token != XContentParser.Token.END_ARRAY) {
                    GeoPoint point = GeoUtils.parseGeoPoint(parser);
                    contexts.add(Geohash.stringEncode(point.getLon(), point.getLat(), this.precision));
                    token = parser.nextToken();
                }
            }
            return contexts;
        } else if (token == XContentParser.Token.VALUE_STRING) {
            String geoHash = parser.text();
            CharSequence truncatedGeoHash = geoHash.subSequence(0, Math.min(geoHash.length(), this.precision));
            contexts.add(truncatedGeoHash.toString());
            return contexts;
        } else {
            GeoPoint point = GeoUtils.parseGeoPoint(parser);
            contexts.add(Geohash.stringEncode(point.getLon(), point.getLat(), this.precision));
        }
        return contexts;
    }

    @Override
    public Set<String> parseContext(LuceneDocument document) {
        HashSet<String> geohashes = new HashSet<String>();
        if (this.fieldName != null) {
            List<IndexableField> fields = document.getFields(this.fieldName);
            GeoPoint spare = new GeoPoint();
            for (IndexableField field : fields) {
                if (field instanceof StringField) {
                    spare.resetFromString(field.stringValue());
                    geohashes.add(spare.geohash());
                    continue;
                }
                if (field instanceof LatLonDocValuesField || field instanceof GeoPointFieldMapper.LatLonPointWithDocValues) {
                    spare.resetFromEncoded(field.numericValue().longValue());
                    geohashes.add(spare.geohash());
                    continue;
                }
                if (!(field instanceof LatLonPoint)) continue;
                BytesRef bytes = field.binaryValue();
                spare.reset(NumericUtils.sortableBytesToInt(bytes.bytes, bytes.offset), NumericUtils.sortableBytesToInt(bytes.bytes, bytes.offset + 4));
                geohashes.add(spare.geohash());
            }
        }
        HashSet<String> locations = new HashSet<String>();
        for (String geohash : geohashes) {
            int precision = Math.min(this.precision, geohash.length());
            String truncatedGeohash = geohash.substring(0, precision);
            locations.add(truncatedGeohash);
        }
        return locations;
    }

    @Override
    protected GeoQueryContext fromXContent(XContentParser parser) throws IOException {
        return GeoQueryContext.fromXContent(parser);
    }

    @Override
    public List<ContextMapping.InternalQueryContext> toInternalQueryContexts(List<GeoQueryContext> queryContexts) {
        ArrayList<ContextMapping.InternalQueryContext> internalQueryContextList = new ArrayList<ContextMapping.InternalQueryContext>();
        for (GeoQueryContext queryContext : queryContexts) {
            int minPrecision = Math.min(this.precision, queryContext.getPrecision());
            GeoPoint point = queryContext.getGeoPoint();
            HashSet<String> locations = new HashSet<String>();
            String geoHash = Geohash.stringEncode(point.getLon(), point.getLat(), minPrecision);
            locations.add(geoHash);
            if (queryContext.getNeighbours().isEmpty() && geoHash.length() == this.precision) {
                Geohash.addNeighbors(geoHash, locations);
            } else if (!queryContext.getNeighbours().isEmpty()) {
                queryContext.getNeighbours().stream().filter(neighbourPrecision -> neighbourPrecision < geoHash.length()).forEach(neighbourPrecision -> {
                    String truncatedGeoHash = geoHash.substring(0, (int)neighbourPrecision);
                    locations.add(truncatedGeoHash);
                    Geohash.addNeighbors(truncatedGeoHash, locations);
                });
            }
            internalQueryContextList.addAll(locations.stream().map(location -> new ContextMapping.InternalQueryContext((String)location, queryContext.getBoost(), location.length() < this.precision)).toList());
        }
        return internalQueryContextList;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void validateReferences(IndexVersion indexVersionCreated, Function<String, MappedFieldType> fieldResolver) {
        if (this.fieldName == null) return;
        MappedFieldType mappedFieldType = fieldResolver.apply(this.fieldName);
        if (mappedFieldType == null) {
            if (!indexVersionCreated.before(IndexVersions.V_7_0_0)) throw new ElasticsearchParseException("field [{}] referenced in context [{}] is not defined in the mapping", this.fieldName, this.name);
            deprecationLogger.warn(DeprecationCategory.MAPPINGS, "geo_context_mapping", "field [{}] referenced in context [{}] is not defined in the mapping", this.fieldName, this.name);
            return;
        } else {
            if ("geo_point".equals(mappedFieldType.typeName())) return;
            if (!indexVersionCreated.before(IndexVersions.V_7_0_0)) throw new ElasticsearchParseException("field [{}] referenced in context [{}] must be mapped to geo_point, found [{}]", this.fieldName, this.name, mappedFieldType.typeName());
            deprecationLogger.warn(DeprecationCategory.MAPPINGS, "geo_context_mapping", "field [{}] referenced in context [{}] must be mapped to geo_point, found [{}]", this.fieldName, this.name, mappedFieldType.typeName());
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        GeoContextMapping that = (GeoContextMapping)o;
        if (this.precision != that.precision) {
            return false;
        }
        return Objects.equals(this.fieldName, that.fieldName);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), this.precision, this.fieldName);
    }

    public static class Builder
    extends ContextBuilder<GeoContextMapping> {
        private int precision = 6;
        private String fieldName = null;

        public Builder(String name) {
            super(name);
        }

        public Builder precision(String precision) {
            return this.precision(DistanceUnit.parse(precision, DistanceUnit.METERS, DistanceUnit.METERS));
        }

        public Builder precision(double meters) {
            int level = GeoUtils.geoHashLevelsForPrecision(meters);
            if (GeoUtils.geoHashCellSize(level) < meters) {
                level = Math.max(1, level - 1);
            }
            return this.precision(level);
        }

        public Builder precision(int level) {
            this.precision = level;
            return this;
        }

        public Builder field(String fieldName) {
            this.fieldName = fieldName;
            return this;
        }

        @Override
        public GeoContextMapping build() {
            return new GeoContextMapping(this.name, this.fieldName, this.precision);
        }
    }
}

