"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.assertRuleUpgradePreview = assertRuleUpgradePreview;
var _react = require("@testing-library/react");
var _detection_engine = require("../../../../../../../../common/api/detection_engine");
var _rule_upgrade_helpers = require("./rule_upgrade_helpers");
var _rule_upgrade_flyout = require("./rule_upgrade_flyout");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function assertRuleUpgradePreview({
  ruleType,
  fieldName,
  humanizedFieldName,
  fieldVersions: {
    initial,
    customized,
    upgrade
  }
}) {
  describe('preview rule upgrade', () => {
    it('previews non-customized field w/ an upgrade (AAB)', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: initial,
          target: upgrade,
          merged: upgrade
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.StockValueCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NONE
      });
      const {
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      expectFieldUpgradeState(fieldUpgradeWrapper, {
        humanizedFieldName,
        upgradeStateSummary: 'No conflicts',
        upgradeStateBadge: 'Ready for update',
        isModified: false
      });
      (0, _rule_upgrade_helpers.toggleFieldAccordion)(fieldUpgradeWrapper);
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-comparisonSide`)).toBeVisible();
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-finalSide`)).toBeVisible();
    });
    it('previews customized field w/o an upgrade (ABA)', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueNoUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NONE
      });
      const {
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      expectFieldUpgradeState(fieldUpgradeWrapper, {
        humanizedFieldName,
        upgradeStateSummary: 'No update',
        isModified: true
      });
      (0, _rule_upgrade_helpers.toggleFieldAccordion)(fieldUpgradeWrapper);
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-comparisonSide`)).toBeVisible();
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-finalSide`)).toBeVisible();
    });
    it('previews customized field w/ the matching upgrade (ABB)', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: upgrade,
          target: upgrade,
          merged: upgrade
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueSameUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NONE
      });
      const {
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      expectFieldUpgradeState(fieldUpgradeWrapper, {
        humanizedFieldName,
        upgradeStateSummary: 'Matching update',
        isModified: true
      });
      (0, _rule_upgrade_helpers.toggleFieldAccordion)(fieldUpgradeWrapper);
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-comparisonSide`)).toBeVisible();
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-finalSide`)).toBeVisible();
    });
    it('previews customized field w/ an upgrade resulting in a solvable conflict (ABC)', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.SOLVABLE
      });
      const {
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      expectFieldUpgradeState(fieldUpgradeWrapper, {
        humanizedFieldName,
        upgradeStateSummary: 'Auto-resolved conflict',
        upgradeStateBadge: 'Review required',
        isModified: true
      });
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-comparisonSide`)).toBeVisible();
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-finalSide`)).toBeVisible();
    });
    it('previews customized field w/ an upgrade resulting in a non-solvable conflict (ABC)', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          base: initial,
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.CustomizedValueCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NON_SOLVABLE
      });
      const {
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      expectFieldUpgradeState(fieldUpgradeWrapper, {
        humanizedFieldName,
        upgradeStateSummary: 'Unresolved conflict',
        upgradeStateBadge: 'Action required',
        isModified: true
      });
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-comparisonSide`)).toBeVisible();
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-finalSide`)).toBeVisible();
    });
    it('missing base - previews customized field w/ an upgrade and no conflict (-AB)', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.MissingBaseCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.NONE
      });
      const {
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      expectFieldUpgradeState(fieldUpgradeWrapper, {
        humanizedFieldName,
        upgradeStateSummary: 'No conflict',
        upgradeStateBadge: 'Ready for update',
        isModified: false
      });
      (0, _rule_upgrade_helpers.toggleFieldAccordion)(fieldUpgradeWrapper);
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-comparisonSide`)).toBeVisible();
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-finalSide`)).toBeVisible();
    });
    it('missing base - previews customized field w/ an upgrade resulting in a solvable conflict (-AB)', async () => {
      (0, _rule_upgrade_flyout.mockRuleUpgradeReviewData)({
        ruleType,
        fieldName,
        fieldVersions: {
          current: customized,
          target: upgrade,
          merged: customized
        },
        diffOutcome: _detection_engine.ThreeWayDiffOutcome.MissingBaseCanUpdate,
        conflict: _detection_engine.ThreeWayDiffConflict.SOLVABLE
      });
      const {
        getByTestId
      } = await (0, _rule_upgrade_flyout.renderRuleUpgradeFlyout)();
      const fieldUpgradeWrapper = getByTestId(`${fieldName}-upgradeWrapper`);
      expectFieldUpgradeState(fieldUpgradeWrapper, {
        humanizedFieldName,
        upgradeStateSummary: 'Auto-resolved conflict',
        upgradeStateBadge: 'Review required',
        isModified: false
      });
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-comparisonSide`)).toBeVisible();
      expect((0, _react.within)(fieldUpgradeWrapper).getByTestId(`${fieldName}-finalSide`)).toBeVisible();
    });
  });
}
function expectFieldUpgradeState(wrapper, params) {
  expect(wrapper).toHaveTextContent(params.humanizedFieldName);
  expect(wrapper).toHaveTextContent(params.upgradeStateSummary);
  if (params.upgradeStateBadge) {
    expect((0, _react.within)(wrapper).getByTitle(params.upgradeStateBadge)).toBeVisible();
  }
  if (params.isModified) {
    expect((0, _react.within)(wrapper).getByTitle('Modified')).toBeVisible();
  } else {
    expect((0, _react.within)(wrapper).queryByTitle('Modified')).not.toBeInTheDocument();
  }
}