"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseDateWithDefault = exports.getTimeRangeSettings = exports.getIntervalSettings = exports.DEFAULT_TO_MOMENT = exports.DEFAULT_FROM_MOMENT = void 0;
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _moment = _interopRequireDefault(require("moment"));
var _fp = require("lodash/fp");
var _get_time_range = require("./get_time_range");
var _constants = require("../../../common/constants");
var _kibana = require("../lib/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Defaults for if everything fails including dateMath.parse(DEFAULT_FROM) or dateMath.parse(DEFAULT_TO)
// These should not really be hit unless we are in an extreme buggy state.
const DEFAULT_FROM_MOMENT = exports.DEFAULT_FROM_MOMENT = (0, _moment.default)().subtract(24, 'hours');
const DEFAULT_TO_MOMENT = exports.DEFAULT_TO_MOMENT = (0, _moment.default)();

/**
 * Retrieves timeRange settings to populate filters
 *
 * @param {Boolean} uiSettings Whether to respect the user's UI settings. Defaults to true.
 */
const getTimeRangeSettings = (uiSettings = true) => {
  const timeRange = uiSettings ? _kibana.KibanaServices.get().uiSettings.get(_constants.DEFAULT_APP_TIME_RANGE) : null;
  const fromStr = (0, _fp.isString)(timeRange === null || timeRange === void 0 ? void 0 : timeRange.from) && (timeRange === null || timeRange === void 0 ? void 0 : timeRange.from) || _constants.DEFAULT_FROM;
  const toStr = (0, _fp.isString)(timeRange === null || timeRange === void 0 ? void 0 : timeRange.to) && (timeRange === null || timeRange === void 0 ? void 0 : timeRange.to) || _constants.DEFAULT_TO;
  const from = parseDateWithDefault(fromStr, DEFAULT_FROM_MOMENT).toISOString();
  const to = parseDateWithDefault(toStr, DEFAULT_TO_MOMENT, true).toISOString();
  const socTrends = (0, _get_time_range.getPreviousTimeRange)({
    to,
    from
  });
  return {
    from,
    fromStr,
    to,
    toStr,
    socTrends
  };
};

/**
 * Retrieves refreshInterval settings to populate filters
 *
 * @param {Boolean} uiSettings Whether to respect the user's UI settings. Defaults to true.
 */
exports.getTimeRangeSettings = getTimeRangeSettings;
const getIntervalSettings = (uiSettings = true) => {
  const interval = uiSettings ? _kibana.KibanaServices.get().uiSettings.get(_constants.DEFAULT_APP_REFRESH_INTERVAL) : null;
  const duration = (0, _fp.isNumber)(interval === null || interval === void 0 ? void 0 : interval.value) && (interval === null || interval === void 0 ? void 0 : interval.value) || _constants.DEFAULT_INTERVAL_VALUE;
  const kind = (0, _fp.isBoolean)(interval === null || interval === void 0 ? void 0 : interval.pause) && !(interval !== null && interval !== void 0 && interval.pause) ? 'interval' : _constants.DEFAULT_INTERVAL_TYPE;
  return {
    kind,
    duration
  };
};

/**
 * Parses a date and returns the default if the date string is not valid.
 * @param dateString The date string to parse
 * @param defaultDate The defaultDate if we cannot parse the dateMath
 * @param {Boolean} roundUp should relative timeranges be rounded up or down
 * @param momentInstance A moment instance to use in place of the library's, will use independent locale settings.
 * @param {Date} forceNow A valid date object to use in place of Date.now()
 * @returns The moment of the date time parsed
 */
exports.getIntervalSettings = getIntervalSettings;
const parseDateWithDefault = (dateString, defaultDate, roundUp = false, momentInstance, forceNow) => {
  const date = _datemath.default.parse(dateString, {
    roundUp,
    momentInstance,
    forceNow
  });
  if (date != null && date.isValid()) {
    return date;
  } else {
    return defaultDate;
  }
};
exports.parseDateWithDefault = parseDateWithDefault;