"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useAwsCredentialsForm = void 0;
var _react = require("react");
var _utils = require("../utils");
var _aws_credentials_form_options = require("./aws_credentials_form_options");
var _constants = require("./constants");
var _constants2 = require("../../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Update CloudFormation template and stack name in the Agent Policy
 * based on the selected policy template
 */

const getSetupFormatFromInput = (input, hasCloudFormationTemplate) => {
  const credentialsType = (0, _utils.getAwsCredentialsType)(input);
  // CloudFormation is the default setup format if the integration has a CloudFormation template
  if (!credentialsType && hasCloudFormationTemplate) {
    return _constants.AWS_SETUP_FORMAT.CLOUD_FORMATION;
  }
  if (credentialsType !== _constants.AWS_CREDENTIALS_TYPE.CLOUD_FORMATION) {
    return _constants.AWS_SETUP_FORMAT.MANUAL;
  }
  return _constants.AWS_SETUP_FORMAT.CLOUD_FORMATION;
};
const useAwsCredentialsForm = ({
  newPolicy,
  input,
  packageInfo,
  updatePolicy
}) => {
  // We only have a value for 'aws.credentials.type' once the form has mounted.
  // On initial render we don't have that value, so we fall back to the default option.

  const options = (0, _aws_credentials_form_options.getAwsCredentialsFormOptions)();
  const hasCloudFormationTemplate = !!(0, _utils.getAssetCloudFormationDefaultValue)(packageInfo);
  const setupFormat = getSetupFormatFromInput(input, hasCloudFormationTemplate);
  const lastManualCredentialsType = (0, _react.useRef)(undefined);

  // Assumes if the credentials type is not set, the default is CloudFormation
  const awsCredentialsType = (0, _utils.getAwsCredentialsType)(input) || _constants.AWS_SETUP_FORMAT.CLOUD_FORMATION;
  const group = options[awsCredentialsType];
  const fields = (0, _aws_credentials_form_options.getInputVarsFields)(input, group.fields);
  const fieldsSnapshot = (0, _react.useRef)({});
  (0, _react.useEffect)(() => {
    // This should ony set the credentials after the initial render
    if (!(0, _utils.getAwsCredentialsType)(input) && !lastManualCredentialsType.current) {
      updatePolicy({
        ...(0, _utils.getAssetPolicy)(newPolicy, input.type, {
          'aws.credentials.type': {
            value: awsCredentialsType,
            type: 'text'
          }
        })
      });
    }
  }, [awsCredentialsType, input, newPolicy, updatePolicy]);
  const elasticDocLink = _constants2.assetIntegrationDocsNavigation.awsGetStartedPath;
  useCloudFormationTemplate({
    packageInfo,
    newPolicy,
    updatePolicy,
    setupFormat
  });
  const onSetupFormatChange = newSetupFormat => {
    if (newSetupFormat === _constants.AWS_SETUP_FORMAT.CLOUD_FORMATION) {
      // We need to store the current manual fields to restore them later
      fieldsSnapshot.current = Object.fromEntries(fields.map(field => [field.id, {
        value: field.value
      }]));
      // We need to store the last manual credentials type to restore it later
      lastManualCredentialsType.current = (0, _utils.getAwsCredentialsType)(input);
      updatePolicy((0, _utils.getAssetPolicy)(newPolicy, input.type, {
        'aws.credentials.type': {
          value: _constants.AWS_CREDENTIALS_TYPE.CLOUD_FORMATION,
          type: 'text'
        },
        // Clearing fields from previous setup format to prevent exposing credentials
        // when switching from manual to cloud formation
        ...Object.fromEntries(fields.map(field => [field.id, {
          value: undefined
        }]))
      }));
    } else {
      updatePolicy((0, _utils.getAssetPolicy)(newPolicy, input.type, {
        'aws.credentials.type': {
          // Restoring last manual credentials type or defaulting to the first option
          value: lastManualCredentialsType.current || _aws_credentials_form_options.DEFAULT_MANUAL_AWS_CREDENTIALS_TYPE,
          type: 'text'
        },
        // Restoring fields from manual setup format if any
        ...fieldsSnapshot.current
      }));
    }
  };
  return {
    awsCredentialsType,
    setupFormat,
    group,
    fields,
    elasticDocLink,
    hasCloudFormationTemplate,
    onSetupFormatChange
  };
};
exports.useAwsCredentialsForm = useAwsCredentialsForm;
const getAwsCloudFormationTemplate = newPolicy => {
  var _newPolicy$inputs, _newPolicy$inputs$fin, _newPolicy$inputs$fin2, _newPolicy$inputs$fin3;
  const template = newPolicy === null || newPolicy === void 0 ? void 0 : (_newPolicy$inputs = newPolicy.inputs) === null || _newPolicy$inputs === void 0 ? void 0 : (_newPolicy$inputs$fin = _newPolicy$inputs.find(i => i.type === _constants.CLOUDBEAT_AWS)) === null || _newPolicy$inputs$fin === void 0 ? void 0 : (_newPolicy$inputs$fin2 = _newPolicy$inputs$fin.config) === null || _newPolicy$inputs$fin2 === void 0 ? void 0 : (_newPolicy$inputs$fin3 = _newPolicy$inputs$fin2.cloud_formation_template_url) === null || _newPolicy$inputs$fin3 === void 0 ? void 0 : _newPolicy$inputs$fin3.value;
  return template || undefined;
};
const updateCloudFormationPolicyTemplate = (newPolicy, updatePolicy, templateUrl) => {
  updatePolicy === null || updatePolicy === void 0 ? void 0 : updatePolicy({
    ...newPolicy,
    inputs: newPolicy.inputs.map(input => {
      if (input.type === _constants.CLOUDBEAT_AWS) {
        return {
          ...input,
          config: {
            cloud_formation_template_url: {
              value: templateUrl
            }
          }
        };
      }
      return input;
    })
  });
};
const useCloudFormationTemplate = ({
  packageInfo,
  newPolicy,
  updatePolicy,
  setupFormat
}) => {
  var _newPolicy$vars;
  (0, _react.useEffect)(() => {
    const policyInputCloudFormationTemplate = getAwsCloudFormationTemplate(newPolicy);
    if (setupFormat === _constants.AWS_SETUP_FORMAT.MANUAL) {
      if (policyInputCloudFormationTemplate) {
        updateCloudFormationPolicyTemplate(newPolicy, updatePolicy, undefined);
      }
      return;
    }
    const templateUrl = (0, _utils.getAssetCloudFormationDefaultValue)(packageInfo);

    // If the template is not available, do not update the policy
    if (templateUrl === '') return;

    // If the template is already set, do not update the policy
    if (policyInputCloudFormationTemplate === templateUrl) return;
    updateCloudFormationPolicyTemplate(newPolicy, updatePolicy, templateUrl);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [newPolicy === null || newPolicy === void 0 ? void 0 : (_newPolicy$vars = newPolicy.vars) === null || _newPolicy$vars === void 0 ? void 0 : _newPolicy$vars.cloud_formation_template_url, newPolicy, packageInfo, setupFormat]);
};