"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suppressAlertsInMemory = exports.isExistingDateGtEqThanAlert = exports.getUpdatedSuppressionBoundaries = exports.createPersistenceRuleTypeWrapper = exports.ALERT_GROUP_INDEX = void 0;
var _lodash = require("lodash");
var _datemath = _interopRequireDefault(require("@elastic/datemath"));
var _server = require("@kbn/alerting-plugin/server");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _fp = require("lodash/fp");
var _get_common_alert_fields = require("./get_common_alert_fields");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Alerts returned from BE have date type coerced to ISO strings
 *
 * We use BackendAlertWithSuppressionFields870 explicitly here as the type instead of
 * AlertWithSuppressionFieldsLatest since we're reading alerts rather than writing,
 * so future versions of Kibana may read 8.7.0 version alerts and need to update them
 */

const ALERT_GROUP_INDEX = exports.ALERT_GROUP_INDEX = `${_ruleDataUtils.ALERT_NAMESPACE}.group.index`;
const augmentAlerts = async ({
  alerts,
  options,
  kibanaVersion,
  currentTimeOverride,
  dangerouslyCreateAlertsInAllSpaces
}) => {
  const commonRuleFields = (0, _get_common_alert_fields.getCommonAlertFields)(options, dangerouslyCreateAlertsInAllSpaces);
  const maintenanceWindowIds = alerts.length > 0 ? await options.services.getMaintenanceWindowIds() : [];
  const currentDate = new Date();
  const timestampOverrideOrCurrent = currentTimeOverride !== null && currentTimeOverride !== void 0 ? currentTimeOverride : currentDate;
  return alerts.map(alert => {
    return {
      ...alert,
      _source: {
        [_ruleDataUtils.ALERT_RULE_EXECUTION_TIMESTAMP]: currentDate,
        [_ruleDataUtils.ALERT_START]: timestampOverrideOrCurrent,
        [_ruleDataUtils.ALERT_LAST_DETECTED]: timestampOverrideOrCurrent,
        [_ruleDataUtils.VERSION]: kibanaVersion,
        ...(maintenanceWindowIds.length ? {
          [_ruleDataUtils.ALERT_MAINTENANCE_WINDOW_IDS]: maintenanceWindowIds
        } : {}),
        ...commonRuleFields,
        ...alert._source
      }
    };
  });
};
const mapAlertsToBulkCreate = alerts => {
  return alerts.flatMap(alert => [{
    create: {
      _id: alert._id
    }
  }, alert._source]);
};

/**
 * finds if any of alerts has duplicate and filter them out
 */
const filterDuplicateAlerts = async ({
  alerts,
  spaceId,
  ruleDataClient
}) => {
  const CHUNK_SIZE = 10000;
  const alertChunks = (0, _lodash.chunk)(alerts, CHUNK_SIZE);
  const filteredAlerts = [];
  for (const alertChunk of alertChunks) {
    const request = {
      query: {
        ids: {
          values: alertChunk.map(alert => alert._id)
        }
      },
      aggs: {
        uuids: {
          terms: {
            field: _ruleDataUtils.ALERT_UUID,
            size: CHUNK_SIZE
          }
        }
      },
      size: 0
    };
    const response = await ruleDataClient.getReader({
      namespace: spaceId
    }).search(request);
    const uuidsMap = {};
    const aggs = response.aggregations;
    if (aggs != null) {
      aggs.uuids.buckets.forEach(bucket => uuidsMap[bucket.key] = true);
      const newAlerts = alertChunk.filter(alert => !uuidsMap[alert._id]);
      filteredAlerts.push(...newAlerts);
    } else {
      filteredAlerts.push(...alertChunk);
    }
  }
  return filteredAlerts;
};

/**
 * suppress alerts by ALERT_INSTANCE_ID in memory
 */
const suppressAlertsInMemory = alerts => {
  const idsMap = {};
  const suppressedAlerts = [];
  const filteredAlerts = (0, _lodash.sortBy)(alerts, alert => alert._source[_ruleDataUtils.ALERT_SUPPRESSION_START]).filter(alert => {
    const instanceId = alert._source[_ruleDataUtils.ALERT_INSTANCE_ID];
    const suppressionDocsCount = alert._source[_ruleDataUtils.ALERT_SUPPRESSION_DOCS_COUNT];
    const suppressionEnd = alert._source[_ruleDataUtils.ALERT_SUPPRESSION_END];
    if (instanceId && idsMap[instanceId] != null) {
      idsMap[instanceId].count += suppressionDocsCount + 1;
      // store the max value of suppression end boundary
      if (suppressionEnd > idsMap[instanceId].suppressionEnd) {
        idsMap[instanceId].suppressionEnd = suppressionEnd;
      }
      suppressedAlerts.push(alert);
      return false;
    } else {
      idsMap[instanceId] = {
        count: suppressionDocsCount,
        suppressionEnd
      };
      return true;
    }
  }, []);
  const alertCandidates = filteredAlerts.map(alert => {
    const instanceId = alert._source[_ruleDataUtils.ALERT_INSTANCE_ID];
    if (instanceId) {
      alert._source[_ruleDataUtils.ALERT_SUPPRESSION_DOCS_COUNT] = idsMap[instanceId].count;
      alert._source[_ruleDataUtils.ALERT_SUPPRESSION_END] = idsMap[instanceId].suppressionEnd;
    }
    return alert;
  });
  return {
    alertCandidates,
    suppressedAlerts
  };
};

/**
 * Compare existing alert suppression date props with alert to suppressed alert values
 **/
exports.suppressAlertsInMemory = suppressAlertsInMemory;
const isExistingDateGtEqThanAlert = (existingAlert, alert, property) => {
  var _existingAlert$_sourc;
  const existingDate = existingAlert === null || existingAlert === void 0 ? void 0 : (_existingAlert$_sourc = existingAlert._source) === null || _existingAlert$_sourc === void 0 ? void 0 : _existingAlert$_sourc[property];
  return existingDate ? existingDate >= alert._source[property].toISOString() : false;
};
exports.isExistingDateGtEqThanAlert = isExistingDateGtEqThanAlert;
/**
 * returns updated suppression time boundaries
 */
const getUpdatedSuppressionBoundaries = (existingAlert, alert, executionId) => {
  var _existingAlert$_sourc2;
  const boundaries = {};
  if (!isExistingDateGtEqThanAlert(existingAlert, alert, _ruleDataUtils.ALERT_SUPPRESSION_END)) {
    boundaries[_ruleDataUtils.ALERT_SUPPRESSION_END] = alert._source[_ruleDataUtils.ALERT_SUPPRESSION_END];
  }
  // start date can only be updated for alert created in the same rule execution
  // it can happen when alert was created in first bulk created, but some of the alerts can be suppressed in the next bulk create request
  if ((existingAlert === null || existingAlert === void 0 ? void 0 : (_existingAlert$_sourc2 = existingAlert._source) === null || _existingAlert$_sourc2 === void 0 ? void 0 : _existingAlert$_sourc2[_ruleDataUtils.ALERT_RULE_EXECUTION_UUID]) === executionId && isExistingDateGtEqThanAlert(existingAlert, alert, _ruleDataUtils.ALERT_SUPPRESSION_START)) {
    boundaries[_ruleDataUtils.ALERT_SUPPRESSION_START] = alert._source[_ruleDataUtils.ALERT_SUPPRESSION_START];
  }
  return boundaries;
};
exports.getUpdatedSuppressionBoundaries = getUpdatedSuppressionBoundaries;
const createPersistenceRuleTypeWrapper = ({
  logger,
  ruleDataClient,
  formatAlert
}) => type => {
  const createAlertsInAllSpaces = (0, _server.shouldCreateAlertsInAllSpaces)({
    ruleTypeId: type.id,
    ruleTypeAlertDef: type.alerts,
    logger
  });
  return {
    ...type,
    executor: async options => {
      const result = await type.executor({
        ...options,
        services: {
          ...options.services,
          alertWithPersistence: async (alerts, refresh, maxAlerts = undefined, enrichAlerts) => {
            const numAlerts = alerts.length;
            logger.debug(`Found ${numAlerts} alerts.`);
            const ruleDataClientWriter = await ruleDataClient.getWriter({
              namespace: options.spaceId
            });

            // Only write alerts if:
            // - writing is enabled
            //   AND
            //   - rule execution has not been cancelled due to timeout
            //     OR
            //   - if execution has been cancelled due to timeout, if feature flags are configured to write alerts anyway
            const writeAlerts = ruleDataClient.isWriteEnabled() && options.services.shouldWriteAlerts();
            if (writeAlerts && numAlerts) {
              const filteredAlerts = await filterDuplicateAlerts({
                alerts,
                ruleDataClient,
                spaceId: options.spaceId
              });
              if (filteredAlerts.length === 0) {
                return {
                  createdAlerts: [],
                  errors: {},
                  alertsWereTruncated: false
                };
              } else if (maxAlerts === 0) {
                return {
                  createdAlerts: [],
                  errors: {},
                  alertsWereTruncated: true
                };
              }
              let enrichedAlerts = filteredAlerts;
              if (enrichAlerts) {
                try {
                  enrichedAlerts = await enrichAlerts(filteredAlerts, {
                    spaceId: options.spaceId
                  });
                } catch (e) {
                  logger.debug('Enrichments failed');
                }
              }
              let alertsWereTruncated = false;
              if (maxAlerts && enrichedAlerts.length > maxAlerts) {
                enrichedAlerts.length = maxAlerts;
                alertsWereTruncated = true;
              }
              const augmentedAlerts = await augmentAlerts({
                alerts: enrichedAlerts,
                options,
                kibanaVersion: ruleDataClient.kibanaVersion,
                currentTimeOverride: undefined,
                dangerouslyCreateAlertsInAllSpaces: createAlertsInAllSpaces
              });
              const response = await ruleDataClientWriter.bulk({
                body: mapAlertsToBulkCreate(augmentedAlerts),
                refresh
              });
              if (response == null) {
                return {
                  createdAlerts: [],
                  errors: {},
                  alertsWereTruncated
                };
              }
              const createdAlerts = augmentedAlerts.map((alert, idx) => {
                var _responseItem$_id, _responseItem$_index;
                const responseItem = response.body.items[idx].create;
                return {
                  _id: (_responseItem$_id = responseItem === null || responseItem === void 0 ? void 0 : responseItem._id) !== null && _responseItem$_id !== void 0 ? _responseItem$_id : '',
                  _index: (_responseItem$_index = responseItem === null || responseItem === void 0 ? void 0 : responseItem._index) !== null && _responseItem$_index !== void 0 ? _responseItem$_index : '',
                  ...alert._source
                };
              }).filter((_, idx) => {
                var _response$body$items$;
                return ((_response$body$items$ = response.body.items[idx].create) === null || _response$body$items$ === void 0 ? void 0 : _response$body$items$.status) === 201;
              })
              // Security solution's EQL rule consists of building block alerts which should be filtered out.
              // Building block alerts have additional "kibana.alert.group.index" attribute which is absent for the root alert.
              .filter(alert => !Object.keys(alert).includes(ALERT_GROUP_INDEX));
              createdAlerts.forEach(alert => {
                var _type$getViewInAppRel, _formatAlert;
                return options.services.alertFactory.create(alert._id).replaceState({
                  signals_count: 1
                }).scheduleActions(type.defaultActionGroupId, {
                  rule: (0, _fp.mapKeys)(_fp.snakeCase, {
                    ...options.params,
                    name: options.rule.name,
                    id: options.rule.id
                  }),
                  results_link: (_type$getViewInAppRel = type.getViewInAppRelativeUrl) === null || _type$getViewInAppRel === void 0 ? void 0 : _type$getViewInAppRel.call(type, {
                    rule: {
                      ...options.rule,
                      params: options.params
                    },
                    start: Date.parse(alert[_ruleDataUtils.TIMESTAMP]),
                    end: Date.parse(alert[_ruleDataUtils.TIMESTAMP])
                  }),
                  alerts: [(_formatAlert = formatAlert === null || formatAlert === void 0 ? void 0 : formatAlert(alert)) !== null && _formatAlert !== void 0 ? _formatAlert : alert]
                });
              });
              return {
                createdAlerts,
                errors: (0, _utils.errorAggregator)(response.body, [409]),
                alertsWereTruncated
              };
            } else {
              logger.debug('Writing is disabled.');
              return {
                createdAlerts: [],
                errors: {},
                alertsWereTruncated: false
              };
            }
          },
          alertWithSuppression: async (alerts, suppressionWindow, enrichAlerts, currentTimeOverride, isRuleExecutionOnly, maxAlerts) => {
            const ruleDataClientWriter = await ruleDataClient.getWriter({
              namespace: options.spaceId
            });

            // Only write alerts if:
            // - writing is enabled
            //   AND
            //   - rule execution has not been cancelled due to timeout
            //     OR
            //   - if execution has been cancelled due to timeout, if feature flags are configured to write alerts anyway
            const writeAlerts = ruleDataClient.isWriteEnabled() && options.services.shouldWriteAlerts();
            let alertsWereTruncated = false;
            if (writeAlerts && alerts.length > 0) {
              const suppressionWindowStart = _datemath.default.parse(suppressionWindow, {
                forceNow: currentTimeOverride
              });
              if (!suppressionWindowStart) {
                throw new Error('Failed to parse suppression window');
              }
              const filteredDuplicates = await filterDuplicateAlerts({
                alerts,
                ruleDataClient,
                spaceId: options.spaceId
              });
              if (filteredDuplicates.length === 0) {
                return {
                  createdAlerts: [],
                  errors: {},
                  suppressedAlerts: [],
                  alertsWereTruncated
                };
              }
              const suppressionAlertSearchRequest = {
                size: filteredDuplicates.length,
                query: {
                  bool: {
                    filter: [{
                      range: {
                        [_ruleDataUtils.ALERT_START]: {
                          gte: suppressionWindowStart.toISOString()
                        }
                      }
                    }, {
                      terms: {
                        [_ruleDataUtils.ALERT_INSTANCE_ID]: filteredDuplicates.map(alert => alert._source[_ruleDataUtils.ALERT_INSTANCE_ID])
                      }
                    }, {
                      bool: {
                        must_not: {
                          term: {
                            [_ruleDataUtils.ALERT_WORKFLOW_STATUS]: 'closed'
                          }
                        }
                      }
                    }]
                  }
                },
                collapse: {
                  field: _ruleDataUtils.ALERT_INSTANCE_ID
                },
                sort: [{
                  [_ruleDataUtils.ALERT_START]: {
                    order: 'desc'
                  }
                }]
              };
              const response = await ruleDataClient.getReader({
                namespace: options.spaceId
              }).search(suppressionAlertSearchRequest);
              const existingAlertsByInstanceId = response.hits.hits.reduce((acc, hit) => {
                acc[hit._source[_ruleDataUtils.ALERT_INSTANCE_ID]] = hit;
                return acc;
              }, {});

              // filter out alerts that were already suppressed
              // alert was suppressed if its suppression ends is older
              // than suppression end of existing alert
              // if existing alert was created earlier during the same
              // rule execution - then alerts can be counted as not suppressed yet
              // as they are processed for the first time against this existing alert
              const nonSuppressedAlerts = filteredDuplicates.filter(alert => {
                var _existingAlert$_sourc3;
                const existingAlert = existingAlertsByInstanceId[alert._source[_ruleDataUtils.ALERT_INSTANCE_ID]];
                if (!existingAlert || (existingAlert === null || existingAlert === void 0 ? void 0 : (_existingAlert$_sourc3 = existingAlert._source) === null || _existingAlert$_sourc3 === void 0 ? void 0 : _existingAlert$_sourc3[_ruleDataUtils.ALERT_RULE_EXECUTION_UUID]) === options.executionId) {
                  return true;
                }
                return !isExistingDateGtEqThanAlert(existingAlert, alert, _ruleDataUtils.ALERT_SUPPRESSION_END);
              });
              if (nonSuppressedAlerts.length === 0) {
                return {
                  createdAlerts: [],
                  errors: {},
                  suppressedAlerts: [],
                  alertsWereTruncated
                };
              }
              const {
                alertCandidates,
                suppressedAlerts: suppressedInMemoryAlerts
              } = suppressAlertsInMemory(nonSuppressedAlerts);
              const [duplicateAlerts, newAlerts] = (0, _lodash.partition)(alertCandidates, alert => {
                const existingAlert = existingAlertsByInstanceId[alert._source[_ruleDataUtils.ALERT_INSTANCE_ID]];

                // if suppression enabled only on rule execution, we need to suppress alerts only against
                // alert created in the same rule execution. Otherwise, we need to create a new alert to accommodate per rule execution suppression
                if (isRuleExecutionOnly) {
                  var _existingAlert$_sourc4;
                  return (existingAlert === null || existingAlert === void 0 ? void 0 : (_existingAlert$_sourc4 = existingAlert._source) === null || _existingAlert$_sourc4 === void 0 ? void 0 : _existingAlert$_sourc4[_ruleDataUtils.ALERT_RULE_EXECUTION_UUID]) === options.executionId;
                } else {
                  return existingAlert != null;
                }
              });
              const duplicateAlertUpdates = duplicateAlerts.flatMap(alert => {
                var _existingAlert$_sourc5, _existingAlert$_sourc6;
                const existingAlert = existingAlertsByInstanceId[alert._source[_ruleDataUtils.ALERT_INSTANCE_ID]];
                const existingDocsCount = (_existingAlert$_sourc5 = (_existingAlert$_sourc6 = existingAlert._source) === null || _existingAlert$_sourc6 === void 0 ? void 0 : _existingAlert$_sourc6[_ruleDataUtils.ALERT_SUPPRESSION_DOCS_COUNT]) !== null && _existingAlert$_sourc5 !== void 0 ? _existingAlert$_sourc5 : 0;
                return [{
                  update: {
                    _id: existingAlert._id,
                    _index: existingAlert._index,
                    require_alias: false
                  }
                }, {
                  doc: {
                    ...getUpdatedSuppressionBoundaries(existingAlert, alert, options.executionId),
                    [_ruleDataUtils.ALERT_LAST_DETECTED]: currentTimeOverride !== null && currentTimeOverride !== void 0 ? currentTimeOverride : new Date(),
                    [_ruleDataUtils.ALERT_SUPPRESSION_DOCS_COUNT]: existingDocsCount + alert._source[_ruleDataUtils.ALERT_SUPPRESSION_DOCS_COUNT] + 1
                  }
                }];
              });

              // we can now augment and enrich
              // the sub alerts (if any) the same as we would
              // any other newAlert
              let enrichedAlerts = newAlerts.some(newAlert => newAlert.subAlerts != null) ? newAlerts.flatMap(newAlert => {
                const {
                  subAlerts,
                  ...everything
                } = newAlert;
                return [everything, ...(subAlerts !== null && subAlerts !== void 0 ? subAlerts : [])];
              }) : newAlerts;
              if (enrichAlerts) {
                try {
                  enrichedAlerts = await enrichAlerts(enrichedAlerts, {
                    spaceId: options.spaceId
                  });
                } catch (e) {
                  logger.debug('Enrichments failed');
                }
              }
              if (maxAlerts && enrichedAlerts.length > maxAlerts) {
                enrichedAlerts.length = maxAlerts;
                alertsWereTruncated = true;
              }
              const augmentedAlerts = await augmentAlerts({
                alerts: enrichedAlerts,
                options,
                kibanaVersion: ruleDataClient.kibanaVersion,
                currentTimeOverride,
                dangerouslyCreateAlertsInAllSpaces: createAlertsInAllSpaces
              });
              const bulkResponse = await ruleDataClientWriter.bulk({
                body: [...duplicateAlertUpdates, ...mapAlertsToBulkCreate(augmentedAlerts)],
                // On serverless we can force a refresh to we don't wait for the longer refresh interval
                // When too many refresh calls are done in a short period of time, they are throttled by stateless Elasticsearch
                refresh: options.isServerless ? true : 'wait_for'
              });
              if (bulkResponse == null) {
                return {
                  createdAlerts: [],
                  errors: {},
                  suppressedAlerts: [],
                  alertsWereTruncated: false
                };
              }
              const createdAlerts = augmentedAlerts.map((alert, idx) => {
                var _responseItem$_id2, _responseItem$_index2;
                const responseItem = bulkResponse.body.items[idx + duplicateAlerts.length].create;
                return {
                  _id: (_responseItem$_id2 = responseItem === null || responseItem === void 0 ? void 0 : responseItem._id) !== null && _responseItem$_id2 !== void 0 ? _responseItem$_id2 : '',
                  _index: (_responseItem$_index2 = responseItem === null || responseItem === void 0 ? void 0 : responseItem._index) !== null && _responseItem$_index2 !== void 0 ? _responseItem$_index2 : '',
                  ...alert._source
                };
              }).filter((_, idx) => {
                var _bulkResponse$body$it;
                return ((_bulkResponse$body$it = bulkResponse.body.items[idx + duplicateAlerts.length].create) === null || _bulkResponse$body$it === void 0 ? void 0 : _bulkResponse$body$it.status) === 201;
              })
              // Security solution's EQL rule consists of building block alerts which should be filtered out.
              // Building block alerts have additional "kibana.alert.group.index" attribute which is absent for the root alert.
              .filter(alert => !Object.keys(alert).includes(ALERT_GROUP_INDEX));
              createdAlerts.forEach(alert => {
                var _type$getViewInAppRel2, _formatAlert2;
                return options.services.alertFactory.create(alert._id).replaceState({
                  signals_count: 1
                }).scheduleActions(type.defaultActionGroupId, {
                  rule: (0, _fp.mapKeys)(_fp.snakeCase, {
                    ...options.params,
                    name: options.rule.name,
                    id: options.rule.id
                  }),
                  results_link: (_type$getViewInAppRel2 = type.getViewInAppRelativeUrl) === null || _type$getViewInAppRel2 === void 0 ? void 0 : _type$getViewInAppRel2.call(type, {
                    rule: {
                      ...options.rule,
                      params: options.params
                    },
                    start: Date.parse(alert[_ruleDataUtils.TIMESTAMP]),
                    end: Date.parse(alert[_ruleDataUtils.TIMESTAMP])
                  }),
                  alerts: [(_formatAlert2 = formatAlert === null || formatAlert === void 0 ? void 0 : formatAlert(alert)) !== null && _formatAlert2 !== void 0 ? _formatAlert2 : alert]
                });
              });
              return {
                createdAlerts,
                suppressedAlerts: [...duplicateAlerts, ...suppressedInMemoryAlerts],
                errors: (0, _utils.errorAggregator)(bulkResponse.body, [409]),
                alertsWereTruncated
              };
            } else {
              logger.debug('Writing is disabled.');
              return {
                createdAlerts: [],
                errors: {},
                suppressedAlerts: [],
                alertsWereTruncated: false
              };
            }
          }
        }
      });
      return result;
    }
  };
};
exports.createPersistenceRuleTypeWrapper = createPersistenceRuleTypeWrapper;