"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createResearcherAgentGraph = void 0;
var _zod = require("@kbn/zod");
var _langgraph = require("@langchain/langgraph");
var _tools = require("@langchain/core/tools");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _graph = require("../chat/graph");
var _prompts = require("./prompts");
var _backlog = require("./backlog");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const setResearchGoalToolName = 'set_research_goal';
const setResearchGoalTool = () => {
  return new _tools.DynamicStructuredTool({
    name: setResearchGoalToolName,
    description: 'use this tool to set the research goal that will be used for the research',
    schema: _zod.z.object({
      reasoning: _zod.z.string().describe('brief reasoning of how and why you defined this research goal'),
      researchGoal: _zod.z.string().describe('the identified research goal')
    }),
    func: () => {
      throw new Error(`${setResearchGoalToolName} was called and shouldn't have`);
    }
  });
};
const StateAnnotation = _langgraph.Annotation.Root({
  // inputs
  initialMessages: (0, _langgraph.Annotation)({
    reducer: _langgraph.messagesStateReducer,
    default: () => []
  }),
  cycleBudget: (0, _langgraph.Annotation)(),
  // budget in number of cycles
  // internal state
  mainResearchGoal: (0, _langgraph.Annotation)(),
  remainingCycles: (0, _langgraph.Annotation)(),
  actionsQueue: (0, _langgraph.Annotation)({
    reducer: (state, actions) => {
      return actions !== null && actions !== void 0 ? actions : state;
    },
    default: () => []
  }),
  pendingActions: (0, _langgraph.Annotation)({
    reducer: (state, actions) => {
      return actions === 'clear' ? [] : [...state, ...actions];
    },
    default: () => []
  }),
  backlog: (0, _langgraph.Annotation)({
    reducer: (current, next) => {
      return [...current, ...next];
    },
    default: () => []
  }),
  // outputs
  generatedAnswer: (0, _langgraph.Annotation)()
});
const createResearcherAgentGraph = async ({
  chatModel,
  tools,
  logger: log
}) => {
  const stringify = obj => JSON.stringify(obj, null, 2);

  /**
   * Identify the research goal from the current discussion, or ask for additional info if required.
   */
  const identifyResearchGoal = async state => {
    const researchGoalModel = chatModel.bindTools([setResearchGoalTool()]).withConfig({
      tags: ['researcher-identify-research-goal', 'researcher-ask-for-clarification']
    });
    const response = await researchGoalModel.invoke((0, _prompts.getIdentifyResearchGoalPrompt)({
      discussion: state.initialMessages
    }));
    const toolCalls = (0, _langchain.extractToolCalls)(response);
    const textContent = (0, _langchain.extractTextContent)(response);
    log.trace(() => `identifyResearchGoal - textContent: ${textContent} - toolCalls: ${stringify(toolCalls)}`);
    if (toolCalls.length > 0) {
      const {
        researchGoal,
        reasoning
      } = toolCalls[0].args;
      const firstAction = {
        question: researchGoal
      };
      return {
        mainResearchGoal: researchGoal,
        backlog: [{
          researchGoal,
          reasoning
        }],
        actionsQueue: [firstAction],
        remainingCycles: state.cycleBudget
      };
    } else {
      const generatedAnswer = textContent;
      return {
        generatedAnswer,
        remainingCycles: state.cycleBudget
      };
    }
  };
  const evaluateResearchGoal = async state => {
    if (state.generatedAnswer) {
      return '__end__';
    }
    return dispatchActions(state);
  };
  const dispatchActions = async state => {
    return state.actionsQueue.map(action => {
      return new _langgraph.Send('perform_search', {
        ...state,
        subResearchGoal: action
      });
    });
  };
  const performSearch = async state => {
    const nextItem = state.subResearchGoal;
    log.trace(() => `performSearch - nextItem: ${stringify(nextItem)}`);
    const executorAgent = (0, _graph.createAgentGraph)({
      chatModel,
      tools,
      logger: log,
      systemPrompt: ''
    });
    const {
      addedMessages
    } = await executorAgent.invoke({
      initialMessages: (0, _prompts.getExecutionPrompt)({
        currentResearchGoal: nextItem,
        backlog: state.backlog
      })
    }, {
      tags: ['executor_agent'],
      metadata: {
        graphName: 'executor_agent'
      }
    });
    const agentResponse = (0, _langchain.extractTextContent)(addedMessages[addedMessages.length - 1]);
    const actionResult = {
      researchGoal: nextItem.question,
      output: agentResponse
    };
    return {
      pendingActions: [actionResult]
    };
  };
  const collectResults = async state => {
    log.trace(() => `collectResults - pending actions: ${stringify(state.pendingActions.map(action => action.researchGoal))}`);
    return {
      pendingActions: 'clear',
      actionsQueue: [],
      backlog: [...state.pendingActions]
    };
  };
  const reflection = async state => {
    const reflectModel = chatModel.withStructuredOutput(_zod.z.object({
      isSufficient: _zod.z.boolean().describe(`Set to true if the current information fully answers the user question without requiring further research.
           Set to false if any knowledge gaps or unresolved sub-problems remain.`),
      nextQuestions: _zod.z.array(_zod.z.string()).describe(`A list of self-contained, actionable research questions or sub-problems that need to be explored
          further to fully answer the user question. Leave empty if isSufficient is true.`),
      reasoning: _zod.z.string().describe(`Brief internal reasoning explaining why the current information is sufficient or not.
            You may list what was already answered, what gaps exist, or whether decomposition was necessary.
            Use this as your thought process or scratchpad before producing the final output.`)
    })).withConfig({
      tags: ['researcher-reflection']
    });
    const response = await reflectModel.invoke((0, _prompts.getReflectionPrompt)({
      userQuery: state.mainResearchGoal,
      backlog: state.backlog,
      maxFollowUpQuestions: 3,
      remainingCycles: state.remainingCycles - 1
    }));
    log.trace(() => `reflection - remaining cycles: ${state.remainingCycles} - response: ${stringify(response)}`);
    return {
      remainingCycles: state.remainingCycles - 1,
      backlog: [response],
      actionsQueue: [...state.actionsQueue, ...response.nextQuestions.map(nextQuestion => ({
        question: nextQuestion
      }))]
    };
  };
  const evaluateReflection = async state => {
    const remainingCycles = state.remainingCycles;
    const reflectionResult = (0, _backlog.lastReflectionResult)(state.backlog);
    if (reflectionResult.isSufficient || remainingCycles <= 0) {
      return 'answer';
    }
    return dispatchActions(state);
  };
  const answer = async state => {
    const answerModel = chatModel.withConfig({
      tags: ['researcher-answer']
    });
    const response = await answerModel.invoke((0, _prompts.getAnswerPrompt)({
      userQuery: state.mainResearchGoal,
      backlog: state.backlog
    }));
    const generatedAnswer = (0, _langchain.extractTextContent)(response);
    log.trace(() => `answer - response ${stringify(generatedAnswer)}`);
    return {
      generatedAnswer
    };
  };

  // note: the node names are used in the event convertion logic, they should *not* be changed
  const graph = new _langgraph.StateGraph(StateAnnotation)
  // nodes
  .addNode('identify_research_goal', identifyResearchGoal).addNode('perform_search', performSearch).addNode('collect_results', collectResults).addNode('reflection', reflection).addNode('answer', answer)
  // edges
  .addEdge('__start__', 'identify_research_goal').addConditionalEdges('identify_research_goal', evaluateResearchGoal, {
    perform_search: 'perform_search',
    __end__: '__end__'
  }).addEdge('perform_search', 'collect_results').addEdge('collect_results', 'reflection').addConditionalEdges('reflection', evaluateReflection, {
    perform_search: 'perform_search',
    answer: 'answer'
  }).addEdge('answer', '__end__').compile();
  return graph;
};
exports.createResearcherAgentGraph = createResearcherAgentGraph;