"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toolsToLangchain = exports.toolToLangchain = exports.toolIdentifierFromToolCall = exports.createToolIdMappings = void 0;
var _tools = require("@langchain/core/tools");
var _onechatCommon = require("@kbn/onechat-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const toolsToLangchain = async ({
  request,
  tools,
  logger
}) => {
  const allTools = Array.isArray(tools) ? tools : await tools.list({
    request
  });
  const mappings = createToolIdMappings(allTools);
  const reverseMappings = reverseMap(mappings);
  const convertedTools = await Promise.all(allTools.map(tool => {
    const toolId = reverseMappings.get((0, _onechatCommon.toSerializedToolIdentifier)({
      toolId: tool.id,
      providerId: tool.meta.providerId
    }));
    return toolToLangchain({
      tool,
      logger,
      toolId
    });
  }));
  return {
    tools: convertedTools,
    idMappings: mappings
  };
};
exports.toolsToLangchain = toolsToLangchain;
const createToolIdMappings = tools => {
  const toolIds = new Set();
  const mapping = new Map();
  for (const tool of tools) {
    let toolId = tool.id;
    let index = 1;
    while (toolIds.has(toolId)) {
      toolId = `${toolId}_${index++}`;
    }
    toolIds.add(toolId);
    mapping.set(toolId, (0, _onechatCommon.toSerializedToolIdentifier)({
      toolId: tool.id,
      providerId: tool.meta.providerId
    }));
  }
  return mapping;
};
exports.createToolIdMappings = createToolIdMappings;
const toolToLangchain = ({
  tool,
  toolId,
  logger
}) => {
  return (0, _tools.tool)(async input => {
    try {
      const toolReturn = await tool.execute({
        toolParams: input
      });
      return JSON.stringify(toolReturn.result);
    } catch (e) {
      logger.warn(`error calling tool ${tool.id}: ${e.message}`);
      throw e;
    }
  }, {
    name: toolId !== null && toolId !== void 0 ? toolId : tool.id,
    description: tool.description,
    schema: tool.schema,
    metadata: {
      serializedToolId: (0, _onechatCommon.toSerializedToolIdentifier)({
        toolId: tool.id,
        providerId: tool.meta.providerId
      })
    }
  });
};
exports.toolToLangchain = toolToLangchain;
const toolIdentifierFromToolCall = (toolCall, mapping) => {
  var _mapping$get;
  return (0, _onechatCommon.toStructuredToolIdentifier)((_mapping$get = mapping.get(toolCall.toolName)) !== null && _mapping$get !== void 0 ? _mapping$get : {
    toolId: toolCall.toolName,
    providerId: _onechatCommon.unknownToolProviderId
  });
};
exports.toolIdentifierFromToolCall = toolIdentifierFromToolCall;
function reverseMap(map) {
  const reversed = new Map();
  for (const [key, value] of map.entries()) {
    if (reversed.has(value)) {
      throw new Error(`Duplicate value detected while reversing map: ${value}`);
    }
    reversed.set(value, key);
  }
  return reversed;
}