"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RelatedDashboardsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _lodash = require("lodash");
var _common = require("@kbn/dashboard-plugin/common");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RelatedDashboardsClient {
  constructor(logger, dashboardClient, alertsClient, alertId) {
    (0, _defineProperty2.default)(this, "dashboardsById", new Map());
    (0, _defineProperty2.default)(this, "alert", null);
    this.logger = logger;
    this.dashboardClient = dashboardClient;
    this.alertsClient = alertsClient;
    this.alertId = alertId;
  }
  async fetchRelatedDashboards() {
    const [alertDocument] = await Promise.all([this.alertsClient.getAlertById(this.alertId), this.fetchFirst500Dashboards()]);
    this.setAlert(alertDocument);
    const [suggestedDashboards, linkedDashboards] = await Promise.all([this.fetchSuggestedDashboards(), this.getLinkedDashboards()]);
    const filteredSuggestedDashboards = suggestedDashboards.filter(suggested => !linkedDashboards.some(linked => linked.id === suggested.id));
    return {
      suggestedDashboards: filteredSuggestedDashboards.slice(0, 10),
      // limit to 10 suggested dashboards
      linkedDashboards
    };
  }
  setAlert(alert) {
    this.alert = alert;
  }
  checkAlert() {
    if (!this.alert) throw new Error(`Alert with id ${this.alertId} not found. Could not fetch related dashboards.`);
    return this.alert;
  }
  async fetchSuggestedDashboards() {
    const alert = this.checkAlert();
    if (!(0, _helpers.isSuggestedDashboardsValidRuleTypeId)(alert.getRuleTypeId())) return [];
    const allSuggestedDashboards = new Set();
    const relevantDashboardsById = new Map();
    const index = this.getRuleQueryIndex();
    const allRelevantFields = alert.getAllRelevantFields();
    if (index) {
      const {
        dashboards
      } = this.getDashboardsByIndex(index);
      dashboards.forEach(dashboard => allSuggestedDashboards.add(dashboard));
    }
    if (allRelevantFields.length > 0) {
      const {
        dashboards
      } = this.getDashboardsByField(allRelevantFields);
      dashboards.forEach(dashboard => allSuggestedDashboards.add(dashboard));
    }
    allSuggestedDashboards.forEach(dashboard => {
      var _relevantDashboardsBy, _relevantDashboardsBy2;
      const dedupedPanels = this.dedupePanels([...(((_relevantDashboardsBy = relevantDashboardsById.get(dashboard.id)) === null || _relevantDashboardsBy === void 0 ? void 0 : _relevantDashboardsBy.relevantPanels) || []), ...dashboard.relevantPanels]);
      const relevantPanelCount = dedupedPanels.length;
      relevantDashboardsById.set(dashboard.id, {
        ...dashboard,
        matchedBy: {
          ...((_relevantDashboardsBy2 = relevantDashboardsById.get(dashboard.id)) === null || _relevantDashboardsBy2 === void 0 ? void 0 : _relevantDashboardsBy2.matchedBy),
          ...dashboard.matchedBy
        },
        relevantPanelCount,
        relevantPanels: dedupedPanels,
        score: this.getScore(dashboard)
      });
    });
    const sortedDashboards = Array.from(relevantDashboardsById.values()).sort((a, b) => {
      return b.score - a.score;
    });
    return sortedDashboards;
  }
  async fetchDashboards({
    page,
    perPage = 20,
    limit
  }) {
    const dashboards = await this.dashboardClient.search({
      limit: perPage,
      cursor: `${page}`
    });
    const {
      result: {
        hits
      }
    } = dashboards;
    hits.forEach(dashboard => {
      this.dashboardsById.set(dashboard.id, dashboard);
    });
    const fetchedUntil = (page - 1) * perPage + dashboards.result.hits.length;
    if (dashboards.result.pagination.total <= fetchedUntil) {
      return;
    }
    if (limit && fetchedUntil >= limit) {
      return;
    }
    await this.fetchDashboards({
      page: page + 1,
      perPage,
      limit
    });
  }
  async fetchFirst500Dashboards() {
    await this.fetchDashboards({
      page: 1,
      perPage: 500,
      limit: 500
    });
  }
  getDashboardsByIndex(index) {
    const relevantDashboards = [];
    this.dashboardsById.forEach(d => {
      const panels = d.attributes.panels;
      const matchingPanels = this.getPanelsByIndex(index, panels);
      if (matchingPanels.length > 0) {
        this.logger.debug(() => `Found ${matchingPanels.length} panel(s) in dashboard ${d.id} using index ${index}`);
        relevantDashboards.push({
          id: d.id,
          title: d.attributes.title,
          description: d.attributes.description,
          matchedBy: {
            index: [index]
          },
          relevantPanelCount: matchingPanels.length,
          relevantPanels: matchingPanels.map(p => ({
            panel: {
              panelIndex: p.panelIndex || (0, _uuid.v4)(),
              type: p.type,
              panelConfig: p.panelConfig,
              title: p.panelConfig.title
            },
            matchedBy: {
              index: [index]
            }
          })),
          score: 0 // scores are computed when dashboards are deduplicated
        });
      }
    });
    return {
      dashboards: relevantDashboards
    };
  }
  dedupePanels(panels) {
    const uniquePanels = new Map();
    panels.forEach(p => {
      var _uniquePanels$get;
      uniquePanels.set(p.panel.panelIndex, {
        ...p,
        matchedBy: {
          ...((_uniquePanels$get = uniquePanels.get(p.panel.panelIndex)) === null || _uniquePanels$get === void 0 ? void 0 : _uniquePanels$get.matchedBy),
          ...p.matchedBy
        }
      });
    });
    return Array.from(uniquePanels.values());
  }
  getDashboardsByField(fields) {
    const relevantDashboards = [];
    this.dashboardsById.forEach(d => {
      const panels = d.attributes.panels;
      const matchingPanels = this.getPanelsByField(fields, panels);
      const allMatchingFields = new Set(matchingPanels.map(p => Array.from(p.matchingFields)).flat());
      if (matchingPanels.length > 0) {
        this.logger.debug(() => `Found ${matchingPanels.length} panel(s) in dashboard ${d.id} using field(s) ${Array.from(allMatchingFields).toString()}`);
        relevantDashboards.push({
          id: d.id,
          title: d.attributes.title,
          description: d.attributes.description,
          matchedBy: {
            fields: Array.from(allMatchingFields)
          },
          relevantPanelCount: matchingPanels.length,
          relevantPanels: matchingPanels.map(p => ({
            panel: {
              panelIndex: p.panel.panelIndex || (0, _uuid.v4)(),
              type: p.panel.type,
              panelConfig: p.panel.panelConfig,
              title: p.panel.panelConfig.title
            },
            matchedBy: {
              fields: Array.from(p.matchingFields)
            }
          })),
          score: 0 // scores are computed when dashboards are deduplicated
        });
      }
    });
    return {
      dashboards: relevantDashboards
    };
  }
  getPanelsByIndex(index, panels) {
    const panelsByIndex = panels.filter(p => {
      if ((0, _common.isDashboardSection)(p)) return false; // filter out sections
      const panelIndices = this.getPanelIndices(p);
      return panelIndices.has(index);
    }); // filtering with type guard doesn't actually limit type, so need to cast
    return panelsByIndex;
  }
  getPanelsByField(fields, panels) {
    const panelsByField = panels.reduce((acc, p) => {
      if ((0, _common.isDashboardSection)(p)) return acc; // filter out sections
      const panelFields = this.getPanelFields(p);
      const matchingFields = fields.filter(f => panelFields.has(f));
      if (matchingFields.length) {
        acc.push({
          matchingFields: new Set(matchingFields),
          panel: p
        });
      }
      return acc;
    }, []);
    return panelsByField;
  }
  getPanelIndices(panel) {
    const emptyIndicesSet = new Set();
    switch (panel.type) {
      case 'lens':
        const maybeLensAttr = panel.panelConfig.attributes;
        if (this.isLensVizAttributes(maybeLensAttr)) {
          const lensIndices = this.getLensVizIndices(maybeLensAttr);
          return lensIndices;
        }
        return emptyIndicesSet;
      default:
        return emptyIndicesSet;
    }
  }
  getPanelFields(panel) {
    const emptyFieldsSet = new Set();
    switch (panel.type) {
      case 'lens':
        const maybeLensAttr = panel.panelConfig.attributes;
        if (this.isLensVizAttributes(maybeLensAttr)) {
          const lensFields = this.getLensVizFields(maybeLensAttr);
          return lensFields;
        }
        return emptyFieldsSet;
      default:
        return emptyFieldsSet;
    }
  }
  isLensVizAttributes(attributes) {
    if (!attributes) {
      return false;
    }
    return Boolean(attributes) && typeof attributes === 'object' && 'type' in attributes && attributes.type === 'lens';
  }
  getRuleQueryIndex() {
    const alert = this.checkAlert();
    const index = alert.getRuleQueryIndex();
    return index;
  }
  getLensVizIndices(lensAttr) {
    const indices = new Set(lensAttr.references.filter(r => r.name.match(`indexpattern`)).map(reference => reference.id));
    return indices;
  }
  getLensVizFields(lensAttr) {
    var _lensAttr$state$datas;
    const fields = new Set();
    const dataSourceLayers = ((_lensAttr$state$datas = lensAttr.state.datasourceStates.formBased) === null || _lensAttr$state$datas === void 0 ? void 0 : _lensAttr$state$datas.layers) || {};
    Object.values(dataSourceLayers).forEach(ds => {
      const columns = ds.columns;
      Object.values(columns).forEach(col => {
        const hasSourceField = c => c.sourceField !== undefined;
        if (hasSourceField(col)) {
          fields.add(col.sourceField);
        }
      });
    });
    return fields;
  }
  async getLinkedDashboards() {
    var _rule$artifacts;
    const alert = this.checkAlert();
    const ruleId = alert.getRuleId();
    if (!ruleId) {
      throw new Error(`Alert with id ${this.alertId} does not have a rule ID. Could not fetch linked dashboards.`);
    }
    const rule = await this.alertsClient.getRuleById(ruleId);
    if (!rule) {
      throw new Error(`Rule with id ${ruleId} not found. Could not fetch linked dashboards for alert with id ${this.alertId}.`);
    }
    const linkedDashboardsArtifacts = ((_rule$artifacts = rule.artifacts) === null || _rule$artifacts === void 0 ? void 0 : _rule$artifacts.dashboards) || [];
    const linkedDashboards = await this.getLinkedDashboardsByIds(linkedDashboardsArtifacts.map(d => d.id));
    return linkedDashboards;
  }
  async getLinkedDashboardsByIds(ids) {
    const linkedDashboardsResponse = await Promise.all(ids.map(id => this.getLinkedDashboardById(id)));
    return linkedDashboardsResponse.filter(dashboard => Boolean(dashboard));
  }
  async getLinkedDashboardById(id) {
    try {
      const dashboardResponse = await this.dashboardClient.get(id);
      return {
        id: dashboardResponse.result.item.id,
        title: dashboardResponse.result.item.attributes.title,
        matchedBy: {
          linked: true
        },
        description: dashboardResponse.result.item.attributes.description
      };
    } catch (error) {
      if (error.output.statusCode === 404) {
        this.logger.warn(`Linked dashboard with id ${id} not found. Skipping.`);
        return null;
      }
      throw new Error(`Error fetching dashboard with id ${id}: ${error.message || error}`);
    }
  }
  getMatchingFields(dashboard) {
    const matchingFields = new Set();
    // grab all the top level arrays from the matchedBy object via Object.values
    Object.values((0, _lodash.omit)(dashboard.matchedBy, 'linked')).forEach(match => {
      // add the values of each array to the matchingFields set
      match.forEach(value => {
        matchingFields.add(value);
      });
    });
    return Array.from(matchingFields);
  }
  getScore(dashboard) {
    const alert = this.checkAlert();
    const allRelevantFields = alert.getAllRelevantFields();
    const index = this.getRuleQueryIndex();
    const setA = new Set([...allRelevantFields, ...(index ? [index] : [])]);
    const setB = new Set(this.getMatchingFields(dashboard));
    const intersection = new Set([...setA].filter(item => setB.has(item)));
    const union = new Set([...setA, ...setB]);
    return intersection.size / union.size;
  }
}
exports.RelatedDashboardsClient = RelatedDashboardsClient;