"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addAnonymizationData = addAnonymizationData;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _conversation_complete = require("../../../../common/conversation_complete");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Operator to process events with deanonymization data and format messages with deanonymizations.
 * Deanonymizations are matched based on `content`, as the Inference client's anonymization process
 * only emits deanonymizations based on `content`. Message ordering is not reliable because not
 * every MessageAddEvent is guaranteed to have the same messages as input as the source messages.
 *
 * @param allMessages The combined messages to use as a fallback if no deanonymization data is found
 * @returns An Observable that emits a single array of messages with deanonymization data added
 */
function addAnonymizationData(messages) {
  return source$ => {
    // Find the latest event with deanonymization data
    return source$.pipe((0, _rxjs.filter)(event => event.type === _conversation_complete.StreamingChatResponseEventType.MessageAdd && !!(event.deanonymized_input || event.deanonymized_output)), (0, _rxjs.toArray)(), (0, _rxjs.map)(events => {
      const clonedMessages = (0, _lodash.cloneDeep)(messages);
      const messagesByContent = (0, _lodash.keyBy)(clonedMessages.filter(item => !!item.message.content), item => item.message.content);
      for (const event of events) {
        var _event$deanonymized_i, _event$deanonymized_o;
        (_event$deanonymized_i = event.deanonymized_input) === null || _event$deanonymized_i === void 0 ? void 0 : _event$deanonymized_i.forEach(item => {
          const matchingMessage = item.message.content ? messagesByContent[item.message.content] : undefined;
          if (matchingMessage) {
            matchingMessage.message.deanonymizations = item.deanonymizations;
          }
        });
        if ((_event$deanonymized_o = event.deanonymized_output) !== null && _event$deanonymized_o !== void 0 && _event$deanonymized_o.message.content) {
          const matchingMessage = event.deanonymized_output.message.content ? messagesByContent[event.deanonymized_output.message.content] : undefined;
          if (matchingMessage) {
            matchingMessage.message.deanonymizations = event.deanonymized_output.deanonymizations;
          }
        }
      }
      return clonedMessages;
    }), (0, _rxjs.defaultIfEmpty)(messages));
  };
}