"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CHANGES_FUNCTION_NAME = void 0;
exports.registerChangesFunction = registerChangesFunction;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _elasticsearch = require("../../clients/elasticsearch");
var _changes = require("../../../common/functions/changes");
var _get_metric_changes = require("./get_metric_changes");
var _get_log_changes = require("./get_log_changes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CHANGES_FUNCTION_NAME = exports.CHANGES_FUNCTION_NAME = 'changes';
function registerChangesFunction({
  functions,
  resources: {
    logger,
    context: {
      core: corePromise
    }
  },
  pluginsStart,
  scopes
}) {
  if (scopes.includes('observability')) {
    functions.registerFunction({
      name: CHANGES_FUNCTION_NAME,
      description: 'Returns change points like spikes and dips for logs and metrics.',
      parameters: _changes.changesFunctionParameters
    }, async ({
      arguments: {
        start,
        end,
        logs = [],
        metrics = []
      }
    }) => {
      if (logs.length === 0 && metrics.length === 0) {
        throw new Error('No metrics or logs were defined');
      }
      const core = await corePromise;
      const logSourcesService = await pluginsStart.logsDataAccess.services.logSourcesServiceFactory.getLogSourcesService(core.savedObjects.client);
      const logsIndexPattern = await logSourcesService.getFlattenedLogSources();
      const client = (0, _elasticsearch.createElasticsearchClient)({
        client: core.elasticsearch.client.asCurrentUser,
        logger,
        inspect: logger.isLevelEnabled('debug')
      });
      const commonFilters = [{
        range: {
          '@timestamp': {
            gte: start,
            lt: end
          }
        }
      }];
      const dateHistogram = {
        field: '@timestamp',
        buckets: 100
      };
      const [metricChanges, logChanges] = await Promise.all([Promise.all([...metrics.map(async metric => {
        var _metric$groupBy;
        const changes = await (0, _get_metric_changes.getMetricChanges)({
          index: metric.index,
          client,
          filters: [...commonFilters, ...(metric.kqlFilter ? [(0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(metric.kqlFilter))] : [])],
          groupBy: (_metric$groupBy = metric.groupBy) !== null && _metric$groupBy !== void 0 ? _metric$groupBy : [],
          type: metric.type || 'count',
          field: metric.field,
          dateHistogram
        });
        return changes.map(change => ({
          name: metric.name,
          ...change
        }));
      })]), Promise.all([...logs.map(async log => {
        var _log$field;
        const changes = await (0, _get_log_changes.getLogChanges)({
          index: log.index || logsIndexPattern,
          client,
          filters: [...commonFilters, ...(log.kqlFilter ? [(0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(log.kqlFilter))] : [])],
          field: (_log$field = log.field) !== null && _log$field !== void 0 ? _log$field : 'message',
          dateHistogram
        });
        return changes.map(change => ({
          name: log.name,
          ...change
        }));
      })])]);
      const allMetricChanges = (0, _lodash.orderBy)(metricChanges.flat(), [item => 'p_value' in item.changes ? item.changes.p_value : Number.POSITIVE_INFINITY]).slice(0, 25);
      const allMetricChangesWithoutTimeseries = allMetricChanges.flat().map(metricChange => {
        return (0, _lodash.omit)(metricChange, 'over_time');
      });
      const allLogChanges = (0, _lodash.orderBy)(logChanges.flat(), [item => 'p_value' in item.changes ? item.changes.p_value : Number.POSITIVE_INFINITY]).slice(0, 25);
      const allLogChangesWithoutTimeseries = allLogChanges.flat().map(logChange => {
        return (0, _lodash.omit)(logChange, 'over_time');
      });
      return {
        content: {
          description: `For each item, the user can see the type of change, the impact, the timestamp, the trend, and the label.
            Do not regurgitate these results back to the user.
            Instead, focus on the interesting changes, mention possible correlations or root causes, and suggest next steps to the user.
            "indeterminate" means that the system could not detect any changes.`,
          changes: {
            metrics: allMetricChangesWithoutTimeseries,
            logs: allLogChangesWithoutTimeseries
          }
        },
        data: {
          changes: {
            metrics: allMetricChanges,
            logs: allLogChanges
          }
        }
      };
    });
  }
}