"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DeploymentParamsMapper = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const THREADS_MAX_EXPONENT = 5;
/**
 * Class responsible for mapping deployment params between API and UI
 */
class DeploymentParamsMapper {
  /**
   * Gets the min allowed number of allocations.
   * - 0 for serverless and ESS with enabled autoscaling.
   * - 1 otherwise
   * @internal
   */
  get minAllowedNumberOfAllocation() {
    return !this.showNodeInfo || this.cloudInfo.isMlAutoscalingEnabled ? 0 : 1;
  }
  constructor(mlServerLimits, cloudInfo, showNodeInfo, nlpSettings) {
    (0, _defineProperty2.default)(this, "threadingParamsValues", void 0);
    /**
     * vCPUs level breakpoints for cloud cluster with enabled ML autoscaling.
     * TODO resolve dynamically when Control Pane exposes the vCPUs range.
     */
    (0, _defineProperty2.default)(this, "autoscalingVCPUBreakpoints", {
      low: {
        min: this.minAllowedNumberOfAllocation,
        max: 2,
        static: 2
      },
      medium: {
        min: 1,
        max: 32,
        static: 32
      },
      high: {
        min: 1,
        max: 99999,
        static: 128
      }
    });
    /**
     * Default vCPUs level breakpoints for serverless projects.
     * Can be overridden by the project specific settings.
     */
    (0, _defineProperty2.default)(this, "serverlessVCPUBreakpoints", {
      low: {
        min: this.minAllowedNumberOfAllocation,
        max: 2,
        static: 2
      },
      medium: {
        min: 0,
        max: 32,
        static: 32
      },
      high: {
        min: 0,
        max: 512,
        static: 512
      }
    });
    /**
     * vCPUs level breakpoints based on the ML server limits.
     * Either on-prem or cloud with disabled ML autoscaling.
     */
    (0, _defineProperty2.default)(this, "hardwareVCPUBreakpoints", void 0);
    /**
     * Result vCPUs level breakpoint based on the cluster env.
     */
    (0, _defineProperty2.default)(this, "vCpuBreakpoints", void 0);
    this.mlServerLimits = mlServerLimits;
    this.cloudInfo = cloudInfo;
    this.showNodeInfo = showNodeInfo;
    this.nlpSettings = nlpSettings;
    /**
     * Initial value can be different for serverless and ESS with autoscaling.
     * Also not available with 0 ML active nodes.
     */
    const maxSingleMlNodeProcessors = this.mlServerLimits.max_single_ml_node_processors;
    this.threadingParamsValues = new Array(THREADS_MAX_EXPONENT).fill(null).map((v, i) => Math.pow(2, i)).filter(maxSingleMlNodeProcessors ? v => v <= maxSingleMlNodeProcessors : v => true);
    const mediumValue = this.mlServerLimits.total_ml_processors / 2;
    this.hardwareVCPUBreakpoints = {
      low: {
        min: this.minAllowedNumberOfAllocation,
        max: 2,
        static: 2
      },
      medium: {
        min: Math.min(3, mediumValue),
        max: mediumValue,
        static: mediumValue
      },
      high: {
        min: mediumValue + 1,
        max: this.mlServerLimits.total_ml_processors,
        static: this.mlServerLimits.total_ml_processors
      }
    };
    if (!this.showNodeInfo) {
      var _this$nlpSettings;
      this.vCpuBreakpoints = this.serverlessVCPUBreakpoints;
      if ((_this$nlpSettings = this.nlpSettings) !== null && _this$nlpSettings !== void 0 && _this$nlpSettings.modelDeployment) {
        // Apply project specific overrides
        this.vCpuBreakpoints = this.nlpSettings.modelDeployment.vCPURange;
      }
    } else if (this.cloudInfo.isMlAutoscalingEnabled) {
      this.vCpuBreakpoints = this.autoscalingVCPUBreakpoints;
    } else {
      this.vCpuBreakpoints = this.hardwareVCPUBreakpoints;
    }
  }
  getNumberOfThreads(input) {
    // 1 thread for ingest at all times
    if (input.optimized === 'optimizedForIngest') return 1;
    // for search deployments with low vCPUs level set 2, otherwise max available
    return input.vCPUUsage === 'low' ? 2 : Math.max(...this.threadingParamsValues);
  }

  /**
   * Returns allocation values accounting for the number of threads per allocation.
   * @param params
   * @internal
   */
  getAllocationsParams(params) {
    const threadsPerAllocation = this.getNumberOfThreads(params);
    const levelValues = this.vCpuBreakpoints[params.vCPUUsage];
    const maxValue = Math.floor(levelValues.max / threadsPerAllocation) || 1;
    return {
      number_of_allocations: maxValue,
      min_number_of_allocations: Math.floor(levelValues.min / threadsPerAllocation) || (
      // For serverless env, always allow scale down to 0
      // For other envs, allow scale down to 0 only for "low" vCPU usage
      this.showNodeInfo === false || params.vCPUUsage === 'low' ? this.minAllowedNumberOfAllocation : 1),
      max_number_of_allocations: maxValue
    };
  }

  /**
   * Gets vCPU (virtual CPU) range based on the vCPU usage level
   * @param vCPUUsage
   * @returns
   */
  getVCPURange(vCPUUsage) {
    return this.vCpuBreakpoints[vCPUUsage];
  }

  /**
   * Gets VCU (Virtual Compute Units) range based on the vCPU usage level
   * @param vCPUUsage
   * @returns
   */
  getVCURange(vCPUUsage) {
    // general purpose (c6gd) 1VCU = 1GB RAM / 0.5 vCPU
    // vector optimized (r6gd) 1VCU = 1GB RAM / 0.125 vCPU
    const vCPUBreakpoints = this.vCpuBreakpoints[vCPUUsage];
    return Object.entries(vCPUBreakpoints).reduce((acc, [key, val]) => {
      // as we can't retrieve Search project configuration, we assume that the vector optimized instance is used
      acc[key] = Math.round(val / 0.125);
      return acc;
    }, {});
  }

  /**
   * Maps UI params to the actual start deployment API request
   * @param input
   */
  mapUiToApiDeploymentParams(modelId, input) {
    var _this$nlpSettings2;
    const resultInput = Object.create(input);
    if (!this.showNodeInfo && ((_this$nlpSettings2 = this.nlpSettings) === null || _this$nlpSettings2 === void 0 ? void 0 : _this$nlpSettings2.modelDeployment.allowStaticAllocations) === false) {
      // Enforce adaptive resources for serverless projects with prohibited static allocations
      resultInput.adaptiveResources = true;
    }
    const allocationParams = this.getAllocationsParams(resultInput);
    return {
      modelId,
      deploymentParams: {
        deployment_id: resultInput.deploymentId,
        threads_per_allocation: this.getNumberOfThreads(resultInput),
        priority: 'normal',
        ...(!resultInput.adaptiveResources && {
          number_of_allocations: allocationParams.number_of_allocations
        })
      },
      ...(resultInput.adaptiveResources && {
        adaptiveAllocationsParams: {
          enabled: true,
          min_number_of_allocations: allocationParams.min_number_of_allocations,
          max_number_of_allocations: allocationParams.max_number_of_allocations
        }
      })
    };
  }

  /**
   * Maps deployment params from API to the UI
   * @param input
   */
  mapApiToUiDeploymentParams(input) {
    var _input$adaptive_alloc, _input$threads_per_al, _input$number_of_allo;
    let optimized = 'optimizedForIngest';
    if (input.threads_per_allocation && input.threads_per_allocation > 1) {
      optimized = 'optimizedForSearch';
    }
    const adaptiveResources = !!((_input$adaptive_alloc = input.adaptive_allocations) !== null && _input$adaptive_alloc !== void 0 && _input$adaptive_alloc.enabled);
    const vCPUs = ((_input$threads_per_al = input.threads_per_allocation) !== null && _input$threads_per_al !== void 0 ? _input$threads_per_al : 0) * (adaptiveResources ? input.adaptive_allocations.max_number_of_allocations : (_input$number_of_allo = input.number_of_allocations) !== null && _input$number_of_allo !== void 0 ? _input$number_of_allo : 0);

    // The deployment can be created via API with a number of allocations that do not exactly match our vCPU ranges.
    // In this case, we should find the closest vCPU range that does not exceed the max or static value of the range.
    const [vCPUUsage] = Object.entries(this.vCpuBreakpoints).filter(([, range]) => vCPUs <= (adaptiveResources ? range.max : range.static)).reduce((prev, curr) => {
      const prevValue = adaptiveResources ? prev[1].max : prev[1].static;
      const currValue = adaptiveResources ? curr[1].max : curr[1].static;
      return Math.abs(vCPUs - prevValue) <= Math.abs(vCPUs - currValue) ? prev : curr;
    },
    // in case allocation params exceed the max value of the high range
    ['high', this.vCpuBreakpoints.high]);
    return {
      deploymentId: input.deployment_id,
      optimized,
      adaptiveResources,
      vCPUUsage: vCPUUsage
    };
  }
}
exports.DeploymentParamsMapper = DeploymentParamsMapper;